# -*- coding: utf-8 -*-

from flask import jsonify, request

from blueflask.lib.errors import bad_request, internal_error
from blueflask.lib.errors import not_found, HTTPError

from ...consts import API_LOGS, SERVICE_CODE
from ... import core
from . import api

import requests


@api.route('/<string:context>/<string:service>', methods=['POST'])
@api.route('/<string:context>/<string:service>/<path:bar>', methods=['POST'])
def call_4D_service(context, service, bar=None):
    """
    Appel générique d'un service 4D.
    Le service appelé est S4D. Le paramètre context qui est passé représente
    le jeu de credentials qu'on doit utiliser.
    ---
    tags:
      - Par défaut
    definitions:
      - schema:
          id: Error
          properties:
            error:
              type: string
              description: summary of the error
            status:
              type: integer
              description: error code
            message:
              type: string
              description: detailed message about the error
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: context
        in: path
        type: string
        description: Le nom du contexte dans lequel se trouve le service web 4D
        required: true
      - name: service
        in: path
        type: string
        description: Le nom du service web 4D à invoquer
        required: true
      - name: bar
        in: path
        type: path
        description: Paramètres supplémentaires pour invoquer le service
        required: false
      - name: data
        in: body
        description: Document JSON contenant les paramètres de la requête
    responses:
      500:
        description: An internal error occured
        $ref: "#/definitions/Error"
      404:
        description: Resource not found
        $ref: "#/definitions/Error"
      200:
        description: Résultat du service 4D
    """
    if bar:
        bar = bar.split('/')
    else:
        bar = []
    data = request.get_json()
    if data.get("version"):
        version = data.get("version")
    else:
        version = '1'
    # trouver un moyen plus safe
    # pour le changement de version
    # soap
    try:
        s4d_request = core.Request(context, service, version, *bar)
        API_LOGS.logger.info(
            "hello from bluebase context {}: service {}".format(
                context,
                service
            )
        )
    except KeyError:
        return not_found(
            message="Le contexte fourni n'existe pas",
            service_code=SERVICE_CODE
            )
    else:
        if data:
            s4d_request.append_data(data)
        try:
            response = s4d_request.send()
        except requests.ConnectionError:
            return not_found(service_code=SERVICE_CODE)
        except requests.HTTPError as exc:
            status = exc.response.status_code
            return HTTPError(status).response()
        except Exception as exc:
            API_LOGS.logger.error(exc)
            return internal_error(
                message=exc,
                service_code=SERVICE_CODE
                )
        else:
            if response['root']['header']['errorNum'] == '0':
                if not response['root'].get('data'):
                    return internal_error(
                        message="La réponse de 4D ne comporte pas de données",
                        service_code=SERVICE_CODE
                        )
                else:
                    error_num = response['root']['data'].get('error_num')
                    if not error_num:
                        response = jsonify(response)
                        return response
                    else:
                        message = response['root']['data']['error']
                        return bad_request(
                            message=message,
                            code='{root_code}-{code}'.format(
                                root_code=SERVICE_CODE,
                                code=error_num
                                ),
                            service_code=SERVICE_CODE
                            )
            else:
                error_num = response['root']['header']['errorNum']
                message = response['root']['header']['errorTx']
                return internal_error(
                    message=message,
                    code='{root_code}-{code}'.format(
                        root_code=SERVICE_CODE,
                        code=error_num
                        ),
                    service_code=SERVICE_CODE
                    )

# EOF
