# -*- coding: utf-8 -*-

"""
Application Flask standard posant les bases d'une API REST
structurée en Blueprints.

Utilise le module blueflask pour standardiser les retours d'erreurs.
Utilise flask_swagger pour créer automatiquement une documentation
OpenAPI à partir des docstrings de chaque endpoint de l'API.
Utilise flask_swagger_ui pour créer une page web affichant la documentation
OpenAPI.
"""

from flask import Flask, g, jsonify, request
from flask_swagger import swagger
from flask_swagger_ui import get_swaggerui_blueprint
from systemd import journal

from .lib.errors import forbidden, internal_error, not_allowed, not_found
from .lib.errors import unauthorized

import uuid


__all__ = [
    'app'
    ]

app = Flask(__name__)
swag_blueprint = get_swaggerui_blueprint('/apidoc', '/api/spec')
app.register_blueprint(swag_blueprint, url_prefix='/apidoc')


@app.route('/api/spec')
def spec():
    """Création de la documentation OpenAPI au format JSON"""
    swag = swagger(app)
    swag['info']['title'] = app.config['TITLE']
    swag['info']['description'] = app.config['DESCRIPTION']
    swag['info']['version'] = app.config['VERSION']
    return jsonify(swag)

# définitions d'erreurs standards pour pallier aux éventuels oublis


@app.errorhandler(401)
def unauthorized_error(e):
    return unauthorized(
        request_id=request.headers.get('X-Request-Id'),
        service_code='040-00-401'
        )


@app.errorhandler(403)
def forbidden_error(e):
    return forbidden(
        request_id=request.headers.get('X-Request-Id'),
        service_code='040-00-403'
        )


@app.errorhandler(404)
def not_found_error(e):
    return not_found(
        request_id=request.headers.get('X-Request-Id'),
        service_code='040-00-404'
        )


@app.errorhandler(405)
def method_not_allowed_error(e):
    return not_allowed(
        request_id=request.headers.get('X-Request-Id'),
        service_code='040-00-405'
        )


@app.errorhandler(500)
def internal_server_error(e):
    return internal_error(
        request_id=request.headers.get('X-Request-Id'),
        service_code='040-00-500'
        )


def actions_before_request():
    """Création d'un callback permettant d'exécuter du code
    avant celui du endpoint appelé"""
    headers = {  # on copie les en-têtes de la requête
        key: value for key, value in request.headers.items()
        }
    if not request.headers.get('X-Request-Id'):
        # si la requête n'a pas d'identifiant on en génère un au hasard
        g.request_id = str(uuid.uuid4())
        headers['X-Request-Id'] = g.request_id
    else:
        g.request_id = request.headers.get('X-Request-Id')
    request.headers = headers
    msg_debug = (
        'Request-Id:{req_id} '
        '{method} '
        '{url} '
        'params={params} '
        'json={json_data}'
        ).format(
            req_id=g.request_id,
            method=request.method,
            url=request.base_url,
            params=request.args or {},
            json_data=request.get_json() or {}
            )
    # on essaye de récupérer le nom de l'utilisateur qui effectue la requête
    try:
        user = g.user
    except Exception:
        user = None
    else:
        if isinstance(user, list):
            user = min(user, key=lambda x: len(x))
    if user:
        msg_debug = 'User:{user} {msg}'.format(user=user, msg=msg_debug)
    journal.send(  # on envoie les détails au journal de systemd
        msg_debug,
        REQUEST_ID=g.request_id
        )


def actions_after_request(response):
    """Création d'un callback permettant d'exécuter du code
    après que la requête soit achevée"""
    response.headers['X-Request-Id'] = g.request_id
    msg_debug = (
        'Request-Id:{req_id} '
        'Status:{status} '
        'Response:{response}').format(
            req_id=g.request_id,
            status=response.status_code,
            response=response.response
            )
    try:
        user = g.user
    except Exception:
        user = None
    else:
        if isinstance(user, list):
            user = min(user, key=lambda x: len(x))
    if user:
        msg_debug = 'User:{user} {msg}'.format(user=user, msg=msg_debug)
    journal.send(
        msg_debug,
        REQUEST_ID=g.request_id,
        STATUS=response.status_code
        )
    return response


app.before_request(actions_before_request)
app.after_request(actions_after_request)


# EOF
