# -*- coding: utf-8  -*-

"""
Collection d'objets flask.jsonify représentant des status
HTTP inférieurs à 400
"""

from flask import jsonify

import os


DEFAULTS = {
    '304': ('aucune modification', "La ressource n'a pas été modifiée"),
    '200': ('ok', 'OK'),
    '201': ('création', 'La ressource a été créée'),
    '202': ('compris', 'La requête a été comprise'),
    '204': None
    }


class HTTPInfo(object):

    def __init__(self, status, service_code=None):
        self.status = str(status)
        self.service_code = (
            service_code or
            os.environ.get('SERVICE_CODE', '04x-yy')
            )

    def response(self, message=None, code=None, request_id=None):
        if self.status == '204':
            response = jsonify({})
        else:
            if not code:
                code = '{root}-{suffix}'.format(
                    root=self.service_code,
                    suffix=self.status
                    )
            response = jsonify(
                {
                    'status': int(self.status),
                    'code': code,
                    'info': DEFAULTS[self.status][0],
                    'message': message or DEFAULTS[self.status][1]
                    }
                )
        response.status_code = int(self.status)
        response.headers = {
            'Content-Type': 'application/json'
            }
        if request_id:
            response.headers['X-Request-Id'] = request_id

        return response


def not_modified(message=None, code=None, request_id=None, service_code=None):
    """HTTP 304"""
    return HTTPInfo(304, service_code).response(message, code, request_id)


def success(message=None, code=None, request_id=None, service_code=None):
    """HTTP 200"""
    return HTTPInfo(200, service_code).response(message, code, request_id)


def created(message=None, code=None, request_id=None, service_code=None):
    """HTTP 201"""
    return HTTPInfo(201, service_code).response(message, code, request_id)


def accepted(message=None, code=None, request_id=None, service_code=None):
    """HTTP 202"""
    return HTTPInfo(202, service_code).response(message, code, request_id)


def no_content(message=None, code=None, request_id=None, service_code=None):
    """HTTP 204"""
    return HTTPInfo(204, service_code).response(message, code, request_id)


# EOF
