# -*- coding: utf-8 -*-

from .consts import ERRORS
from .core import cryptoguard_command
from . import customer_management


__all__ = [
    'add_card',
    'add_customer_definition_channel',
    'add_subscription',
    'blacklist_card',
    'invalidate_card',
    'get_card_informations',
    'list_card',
    'list_stored_cards',
    'list_subscription',
    'new_signal',
    'pair_card',
    'revalidate_card',
    'send_long_message',
    'send_message',
    'set_card_status',
    'start_subscription',
    'stop_subscription',
    'un_pair_card'
    ]


def get_card_informations(cardnumber):
    """Compilation des informations d'une carte"""
    result = list_card(cardnumber)
    code = result['code']
    if code.split('-')[-1] == '0':
        stb = result['message']['stb']
        result = list_subscription(cardnumber)
        code = result['code']
        if code.split('-')[-1] == '0':
            subscriptions = result['message']['subscriptions']
            result = {
                'status': 200,
                'code': code,
                'info': "Informations d'une carte TV",
                'message': {
                    'cardnumber': cardnumber,
                    'stb': stb,
                    'subscriptions': subscriptions
                    }
                }
    return result


def set_card_status(cardnumber, status=True):
    """Activer ou désactiver un carte"""
    customernumber = cardnumber
    if status:  # on assigne à un client ou on active
        result = add_card(cardnumber, customernumber)
        code = result.split('-')[-1]
        if code == '19':  # la carte est déjà assignée à un client
            result = revalidate_card(cardnumber)
            # ce n'est pas gênant de ré-activer une carte déjà assignée
        elif code == '12':  # il n'existe aucun client prêt à porter la carte
            result = customer_management.add_customer(
                cardnumber,
                customernumber,
                '1'
                )
            code = result.split('-')[-1]
            if code == '0':
                result = add_card(cardnumber, customernumber)
    else:  # on désactive la carte
        result = invalidate_card(cardnumber)
    return result


@cryptoguard_command()
def list_stored_cards(**kwargs):
    """Retourne toutes les cartes détruites dans le CAS"""
    result = kwargs['result']
    cards = result['message']['Card']
    cards = [
        {'cardnumber': card['@Number'].replace('-', '')}
        for card in cards
        ]
    result['message'] = cards
    return result


@cryptoguard_command('cardnumber')
def list_card(cardnumber, **kwargs):
    """Retourne les infos d'une carte"""
    result = kwargs['result']
    card = result['message']['Card'][0]
    if int(card['@Stored']):
        result = ERRORS['6']
        result['message']['cardnumber'] = cardnumber
    elif int(card['@Invalid']):
        result = ERRORS['37']
        result['message']['cardnumber'] = cardnumber
    else:
        stb = card['@IRDNumber']
        result['message'] = {
            'cardnumber': cardnumber,
            'stb': stb.zfill(12)
            }
    return result


@cryptoguard_command('cardnumber')
def list_subscription(cardnumber, **kwargs):
    """Retourne les abonnements d'une carte TV"""
    result = kwargs['result']
    subscriptions = result['message']['Subscription']
    subscriptions = [
        {
            'id': subscription['@ArtNr'],
            'name': subscription['@Name'],
            'stop': subscription.get('@EndDate', '')
            }
        for subscription in subscriptions
        ]
    result['message'] = {
        'cardnumber': cardnumber,
        'subscriptions': subscriptions
        }
    return result


@cryptoguard_command('cardnumber', 'startdate', 'enddate', 'artnr')
def add_subscription(cardnumber, startdate, enddate, artnr, **kwargs):
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber,
        'start': startdate,
        'stop': enddate,
        'artnr': artnr
        }
    return result


@cryptoguard_command('cardnumber', 'artnr', 'channelid', 'monthend')
def add_customer_definition_channel(
        cardnumber,
        artnr,
        channelid,
        monthend,
        **kwargs
        ):
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber,
        'artnr': artnr,
        'channelid': channelid,
        'monthend': monthend
        }
    return result


@cryptoguard_command('cardnumber', 'startdate', 'enddate', 'artnr')
def start_subscription(cardnumber, startdate, enddate, artnr, **kwargs):
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber,
        'start': startdate,
        'stop': enddate,
        'artnr': artnr
        }
    return result


@cryptoguard_command('cardnumber', 'stopdate', 'artnr')
def stop_subscription(cardnumber, stopdate, artnr, **kwargs):
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber,
        'stop': stopdate,
        'artnr': artnr
        }
    return result


@cryptoguard_command('cardnumber', 'message')
def send_message(cardnumber, message, **kwargs):
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber,
        'message': message
        }
    return result


@cryptoguard_command(
    'cardnumber',
    'duration',
    'senddate',
    'forced',
    'sid',
    'message'
    )
def send_long_message(
        cardnumber,
        duration,
        senddate,
        forced,
        sid,
        message,
        **kwargs
        ):
    """2048 chars max."""
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber,
        'message': message
        }
    return result


@cryptoguard_command('cardnumber')
def new_signal(cardnumber, **kwargs):
    """Renouvellement des EMMS"""
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber
        }
    return result


@cryptoguard_command('cardnumber')
def un_pair_card(cardnumber, **kwargs):
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber
        }
    return result


@cryptoguard_command('cardnumber', 'irdnumber')
def pair_card(cardnumber, irdnumber, **kwargs):
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber,
        'stb': irdnumber
        }
    return result


@cryptoguard_command('cardnumber', 'customernumber')
def add_card(cardnumber, customernumber, **kwargs):
    """Assigner la carte à un client"""
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber
        }
    return result


@cryptoguard_command('cardnumber')
def revalidate_card(cardnumber, **kwargs):
    """Réactivation de la carte"""
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber
        }
    return result


@cryptoguard_command('cardnumber')
def invalidate_card(cardnumber, **kwargs):
    """Désactivation de la carte"""
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber
        }
    return result


@cryptoguard_command('cardnumber')
def blacklist_card(cardnumber, **kwargs):
    """Destruction de la carte"""
    result = kwargs['result']
    result['message'] = {
        'cardnumber': cardnumber
        }
    return result

# EOF
