# -*- coding: utf-8 -*-

"""
Ce module contient la représentation d'une commande Cryptoguard.
Une commande Cryptoguard est caractérisée par:

  - un nom de méthode

  - un paramètre xmldata représentant la requête à faire vers l'API XML-RPC
"""

from collections import OrderedDict

from rest_client import RestClient

from .consts import CONFIG, ERRORS, SERVICE_CODE

import functools
import xml

import xmltodict


__all__ = [
    'CryptoguardCommand',
    'cryptoguard_command'
    ]


class CryptoguardCommand(object):
    """Représentation d'une commande Cryptoguard"""

    def __init__(self, method, **kwargs):
        self._method = method
        for key, value in kwargs.items():
            setattr(self, key, value)

    def __repr__(self):
        return '<CryptoguardCommand {}>'.format(self._method)

    @property
    def xmldata(self):
        """Représentation de la chaine de caractère XML
        décrivant la méthode à invoquer via l'API XML-RPC"""
        xmldata = {'xmldata': {'@token': CONFIG['VendorAPI']['token']}}
        for key, value in self.__dict__.items():
            if key.startswith('_'):
                xmldata['xmldata']['@'+key[1:]] = value
            else:
                xmldata['xmldata'][key] = value
        return xmltodict.unparse(xmldata).split('\n')[1]

    def send(self):
        """Envoi de la commande à l'API XML-RPC du CAS Cryptoguard"""
        client = RestClient(CONFIG['VendorAPI']['url'])
        response = client.post(
            '/api.php',
            headers={
                'content-type': 'application/x-www-form-urlencoded',
                'method': 'POST'
                },
            data={'xml': self.xmldata}
            )
        if response['status'] == 200:
            xmldata = response['content']
            try:
                xmldata = xmltodict.parse(xmldata)
            except xml.parsers.expat.ExpatError as err:
                message = 'Vendor API returned non XML data'
                result = {
                    'error': 'erreur interne au serveur',
                    'code': '103',
                    'status': 500,
                    'message': message
                    }
                return result
            else:
                if xmldata[self._method]['Status'] != 'OK':
                    status = xmldata[self._method]['ErrorNumber']
                    result = ERRORS[status]
                else:
                    result = {
                        'code': '{}-0'.format(SERVICE_CODE),
                        'info': 'OK',
                        'message': 'OK',
                        'status': 200
                        }
                    del xmldata[self._method]['APIVersion']
                    del xmldata[self._method]['Status']
                    if xmldata[self._method]:
                        data = {}
                        for key, value in xmldata[self._method].items():
                            # value is an OrderedDict or a list of it
                            if isinstance(value, OrderedDict):
                                data[key] = [value]
                            else:
                                data[key] = [v for v in value]
                        result['message'] = data
                return result
        else:
            result = {
                'code': '{}-{}'.format(SERVICE_CODE, response.status_code),
                'error': 'HTTP {}'.format(response.status_code),
                'message': (
                    'https://en.wikipedia.org/wiki'
                    '/List_of_HTTP_status_codes'
                    ),
                'status': response.status_code
                }
            return result


def cryptoguard_command(*command_args):
    """
    Ce décorateur permet:
        - de transformer une fonction en commande Cryptoguard
        - et d'exécuter la commande
    La particularité de ce décorateur est qu'il prend en paramètre
    la signature de la commande Cryptoguard (avec les noms exacts
    des paramètres tels que définis dans la documentation
    l'API XML-RPC.

    Ainsi, si on a ceci:

    @command('cardnumber', 'artnr', 'channelid', 'monthend')
    def f(a, b, c, d, **kwargs):
        pass

    Alors la fonction 'f' est transformée en commande Cryptoguard 'F',
    et les paramètres utilisés pour former le XML
    suivent le mapping ci-dessous:
    {
        'cardnumber': a,
        'artnr': b,
        'channelid': c,
        'monthend': d
        }
    Le résultat de la commande est mis dans deux arguments nommés:

       - raw_result: le résultat tel qu'envoyé par l'API XML-RPC
       - result: le résultat filtré et quelque peu parsé
    Note: les fonctions qui porteront ce décorateur devront donc
    déclarer au moins deux variables nommés prêtes à recevoir
    les résultats (voir exemple).
    """
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            x = len(args)
            if x and x != len(command_args):
                error = (
                    "Il y a une incohérence entre le nombre "
                    "d'arguments déclarés dans le décorateur "
                    "et le nombre d'arguments passés à la fonction."
                    )
                raise SyntaxError(error)
            else:
                bar = dict(zip(command_args, args))
                for key, value in bar.items():
                    kwargs['_'+key] = value

            request = CryptoguardCommand(
                func.__name__.title().replace('_', ''),
                **kwargs
                )
            result = request.send()

            if result['status'] != 200:
                if command_args[0] == 'cardnumber':
                    result['message']['cardnumber'] = args[0]
                return result
            else:
                kwargs['raw_result'] = result
                kwargs['result'] = {
                    'code': result['code'],
                    'status': result['status'],
                    'info': result['info'],
                    'message': []
                    }
                if 'message' not in result:
                    return kwargs['result']
                else:
                    kwargs['result']['message'] = result['message']
                    return func(*args, **kwargs)
        return wrapper
    return decorator

# EOF
