# -*- coding: utf-8 -*-

from datetime import datetime

from systemd import journal

from .consts import CONFIG
from .consts import INFOS
from .consts import PROGRAMS
from . import cluster
from . import media
from .streaming import dtplay, vlc

import collections
import itertools
import os
import shlex
import string
import subprocess
import time

import redis
import requests
import xmltodict


__all__ = ['Program']


def build_tmira_datetime(dt):
    """Créer une datetime au format de epg.blueline.mg.
    Le format d'entrée est %d/%m/%Y %H:%M:%S
    """
    date, hour = dt.split()
    day, month, year = date.split('/')
    result = '{year}-{month}-{day}T{hour}+03:00'.format(
        hour=hour, year=year, month=month, day=day)
    return result


def createtitle(name):
    """Créer un titre avec un nom de fichier media"""
    name = name
    if name.endswith('.ts'):
        name = name[:-3]
    header = name
    name = header.replace('_', ' ').title()
    letters = [
        i for i in string.letters[26:]
        if i not in ['A', 'E', 'I', 'O']
        ]
    for letter in letters:
        if ' '+letter+' ' in name:
            name = name.replace(letter+' ', letter+"'")
    bar = [i for i in name.split("'")[:-1] if len(i) > 1]
    for j in bar:
        name = name.replace(j+"'", j+" ")
    if name.startswith('Ba '):
        name = name.replace('Ba ', 'Bande Annonce ')
    else:
        name = name.replace(' Ctv4', '')
        name = name.replace(' Ctv3', '')
        name = name.replace(' Ctv2', '')
        name = name.replace(' Ctv', '')
    return name


class Program(object):
    """représente une succession de fichiers media
    à jouer pendant une journée, sur une chaine précise
    """

    def __init__(self, day, service_id):
        """Un programme est caractérisé par:
              - une journée: chaine de caractère au format ddmmyyyy
              - un numéro de chaine: un entier
        """
        self.day = day
        self.service_id = service_id
        self.start = CONFIG['SERVICE:{}'.format(self.service_id)]['start']
        self.start_timestamp = time.mktime(
            time.strptime(
                '{0} {1}'.format(self.day, self.start),
                '%d%m%Y %H%M%S'
                )
            )
        self.limit = self.start_timestamp + (24*3600)
        self.redis_zset_key = '{day}:{service_id}'.format(
            day=self.day,
            service_id=self.service_id
            )

    def infos(self, timestamp=None):
        """Retourne le programme sous forme de dicitonnaire.
        Peut aussi renvoyer le programme à partir d'une certaine
        heure précisée par la variable 'timestamp'.
        """
        data = PROGRAMS.zrange(self.redis_zset_key, 0, -1, withscores=True)
        result = {}
        if data:
            for filepath, score in data:
                result[score] = filepath.decode('utf-8')
            timestamps = sorted(result)
            if timestamp:
                if timestamp < timestamps[0]:
                    # si l'heure de début des programmes est supérieure
                    # à l'heure qu'on a passé en paramètre,
                    # on retourne un programme vide
                    return {}
                else:
                    # sinon, on cherche l'heure de début qui vient juste
                    # avant l'heure qu'on a passé en paramètre;
                    # donc on va d'abord ressortir la liste des heures
                    # de début antérieures à l'heure passée en paramètre
                    keys = sorted(
                        [key for key in timestamps if key <= timestamp]
                        )
                    # ensuite on va simplement prendre la dernière heure dans
                    # cette liste: c'est la plus proche de celle qu'on a
                    # passé en paramètre
                    start = keys[-1]
                    result = {
                        key: value
                        for key, value in result.items()
                        if key >= start
                        }
        return result

    def tmira_xmltv(self):
        """Créé un fichier XMLTV à charger sur epg.blueline.mg.
        Les informations sur les synopsis des films sont dans
        une base de données.
        """
        xml = {
            'BroadcastData': {
                'ProviderInfo': {
                    'ProviderId': '',
                    'ProviderName': '',
                    },
                'ScheduleData': {
                    'ChannelPeriod': {
                        'ChannelId': self.service_id,
                        'Event': [],
                        },
                    },
                },
            }
        channelperiod = xml['BroadcastData']['ScheduleData']
        channelperiod = channelperiod['ChannelPeriod']
        infos = self.infos()
        timestamps = sorted(infos)
        for timestamp in timestamps:
            filename = infos[timestamp]
            if not filename.startswith('ba_'):
                # on n'inclue pas les fichiers de bande
                # annonce dans l'EPG
                start = datetime.fromtimestamp(
                    timestamp
                    ).strftime(
                        '%d/%m/%Y %H:%M:%S'
                        )
                filename = filename.split('/')[-1].split(':')[0]
                tsfile = media.Media(filename)
                info = tsfile.details()
                if not info:
                    name = filename.replace('.ts', '')
                    desc = ''
                else:
                    name = info.name
                    desc = info.description
                if '@beginTime' not in channelperiod:
                    channelperiod['@beginTime'] = build_tmira_datetime(start)
                event = {
                    '@duration': str(tsfile.duration),
                    '@beginTime': build_tmira_datetime(start),
                    'EventType': 'S',
                    'FreeAccess': '1',
                    'Unscrambled': '1',
                    'EpgProduction': {
                        'ParentalRating': {
                            '@countryCode': "MDG",
                            '#text': '0',
                            },
                        'DvbContent': {
                            'Content': {
                                '@nibble2': '0',
                                '@nibble1': '0',
                                },
                            },
                        'EpgText': {
                            '@language': 'fre',
                            'ShortDescription': '',
                            'Description': desc,
                            'Name': name,
                            },
                        },
                    }
                channelperiod['Event'].append(event)
        else:
            channelperiod['@endTime'] = build_tmira_datetime(start)
        return xml

    def get_filename_schedule(self, filename):
        """Renvoie le(s) timestamp(s) au(x)quel(s)
        un fichier est censé être diffusé
        """
        occurences = PROGRAMS.zrange(
            self.redis_zset_key,
            0,
            -1,
            withscores=True
            )
        occurences = [
            (filepath.decode('utf-8'), timestamp)
            for filepath, timestamp in occurences
            if filepath.decode('utf-8').split(':')[0].endswith('/'+filename)
            ]
        if not occurences:
            return None
        else:
            result = {
                timestamp: filepath.split(':')[1]
                for filepath, timestamp in occurences
                }
            return result

    def update(self):
        """Mise à jour du programme dans la base de données
        pour corriger éventuellement les durées et les fichiers
        manquants"""
        pass

    def check(self):
        """Routine de vérification de la cohérence d'un programme"""
        answer = {
            'redis': None,
            'program': None,
            'epg': None,
            'coherence': None
            }
        # récupération de l'epg depuis epg.blueline.mg
        url = '{url}/{service_id}/xmls/{date}.xml'.format(
            url='http://epg.blueline.mg/tmira/tmDB/epg/data',
            service_id=self.service_id,
            date=datetime.strptime(self.day, '%d%m%Y').strftime('%Y%m%d')
            )
        try:
            req = requests.get(url, timeout=5)
        except requests.Timeout:
            pass
        else:
            if req.status_code == 200:
                answer['epg'] = True
                epg = xmltodict.parse(req.text)
                data = epg['BroadcastData']['ScheduleData']
                events = data['ChannelPeriod'].get('Event', [])
                if type(events) is collections.OrderedDict:
                    events = [events]
                program_infos = [
                    event['@beginTime'].split('+03:00')[0]
                    for event in events
                    ]
            elif req.status_code == 404:
                answer['epg'] = False
        # récupération de la programmation depuis la base redis
        try:
            PROGRAMS.ping()
        except redis.TimeoutError:
            answer['redis'] = False
        else:
            answer['redis'] = True
            infos = self.infos()
            timestamps = sorted(infos)
            if not infos:
                answer['program'] = False
            else:
                answer['program'] = True
                # on vérifie que la dernière émission se termine
                # avant la limite
                last_program = infos[timestamps[-1]]
                last_media = last_program.split(':')[0].split('/')[-1]
                overflow = media.Media(last_media).duration + timestamps[-1]
                if overflow >= self.limit:
                    answer['overflow'] = overflow
                files = []
                for timestamp, path in infos.items():
                    filepath = path.split(':')[0]
                    filename = filepath.split('/')[-1]
                    if not media.Media(filename).exists:
                        files.append(filename)
                if files:
                    answer['program'] = files
                program = [
                    (j, i) for i, j in infos.items()
                    if not j.split('/')[-1].startswith('ba_')
                    ]
                count = 0
                if program and answer['epg']:
                    for start in program_infos:
                        start = time.mktime(
                            time.strptime(
                                start, '%Y-%m-%dT%H:%M:%S'
                                )
                            )
                        for i, j in program:
                            if abs(start-j) <= 60:
                                count += 1
                                break
                    if count == len(program_infos) and count == len(program):
                        answer['coherence'] = True
                    else:
                        answer['coherence'] = False
        return answer

    def sync(self, remote=None):
        """
        Synchronisation avec le serveur master du cluster si aucun serveur
        distant n'est mentionné.
        Requiert que tous les serveurs REDIS de chaque serveur du cluster
        soient accessibles à distance.
        """
        self_status = INFOS.get('status').decode('utf-8')
        if self_status == 'master':
            journal.send(
                (
                    "Pas de synchronisation parce que "
                    "je suis le serveur en production"
                    ),
                ACTION='Program.sync'
                )
        elif self_status == 'slave':
            if not remote:
                # on synchronise par défaut avec le master du cluster
                server = cluster.Server().get_master()
                if not server:
                    raise SystemError("Aucun serveur master dans le cluster")
            else:
                server = cluster.Server(remote)
            # on synchronise avec 'server'
            # en gros, on le fait à la bourrin: on écrase le local
            PROGRAMS.delete(self.redis_zset_key)
            PEER = redis.Redis(host=server.address, socket_timeout=5)
            infos = PEER.zrange(
                self.redis_zset_key,
                0,
                -1,
                withscores=True
                )
            infos = {
                path.decode('utf-8'): timestamp
                for path, timestamp in infos
                }
            seen = set()
            for path, timestamp in infos.items():
                filepath, index = path.split(':')
                filename = filepath.split('/')[-1]
                m = media.Media(filename)
                PROGRAMS.zadd(
                    self.redis_zset_key,
                    '{}:{}'.format(m.filepath, index),
                    timestamp
                    )
                journal.send(
                    '{}:{} = {}'.format(m.filepath, index, timestamp),
                    ACTION='Program.sync'
                    )
                if not m.exists:
                    if filename not in seen:
                        # on copie le fichier depuis le master
                        journal.send(
                            'Copie du fichier {peer}:{path}'.format(
                                peer=server.hostname,
                                path=filepath
                                ),
                            ACTION='Program.sync'
                            )
                        cmd = (
                            'rsync --progress -z '
                            '{peer}:{path} {dst}'
                            ).format(
                                peer=server.hostname,
                                path=filepath,
                                dst=CONFIG['APP']['media_folder']
                                )
                        subprocess.call(shlex.split(cmd))
                        seen.add(filename)

    def play(self, output, udp_addr=None):
        """Diffusion du programme via DtPlay"""
        now = time.time()
        date_time_now = datetime.fromtimestamp(
            time.time()
            ).strftime('%d%m%Y_%H%M%S')
        infos = self.infos(now)
        timestamps = sorted(infos)
        start = timestamps[0]  # la première heure de diffusion
        diff = now - start
        if diff < 0:
            # nous avons lancé la diffusion à un moment antérieure
            # à la première heure de diffusion; on va donc sagement
            # attendre
            time.sleep(abs(diff))
        elif diff > 0:
            # nous avons lancé la diffusion alors que le premier film
            # était déjà censé être diffusé il y a |diff| secondes
            filepath = infos[start].split(':')[0]
            filename = filepath.split('/')[-1]
            trimmed_file = filepath.replace(
                filename,
                'trimmed/{}_{}'.format(
                    date_time_now,
                    filename
                    )
                )
            # on va donc re-traiter le premier film pour reprendre la diffusion
            # à |diff| + 10 secondes
            # (on estime à 10 secondes le temps de traitement)
            m, s = divmod(diff+10, 60)
            h, m = divmod(m, 60)
            h_m_s = "%d:%02d:%02d" % (h, m, s)
            trim_file = (
                "/usr/bin/avconv -i {0} -ss {1} -codec copy "
                "-mpegts_service_id {2} -mpegts_start_pid {3} "
                "-metadata service_provider='CTV' "
                "-metadata service_name='{4}' "
                "-f mpegts -muxrate 2500k -y {5}").format(
                    filepath,
                    h_m_s,
                    self.service_id,
                    CONFIG['SERVICE:{}'.format(self.service_id)]['pid_video'],
                    CONFIG['SERVICE:{}'.format(self.service_id)]['name'],
                    trimmed_file
                    )
            subprocess.call(shlex.split(trim_file))
        for timestamp in timestamps:
            if timestamp == start:
                movie = trimmed_file
            else:
                movie = infos[timestamp].split(':')[0]
            dtplay(movie, self.service_id, output)
        while time.time() < self.limit:
            # la diffusion est finie mais il reste du temps
            movie = infos[timestamps[-1]].split(':')[0]
            dtplay(movie, self.service_id, output)

    def xspf(self):
        """Création d'une playlist au format XSPF à partir du programme"""
        xspf = {
            'playlist': {
                '@xmlns': 'http://xspf.org/ns/0/',
                '@xmlns:vlc': 'http://www.videolan.org/vlc/playlist/ns/0/',
                '@version': '1'
                }
            }
        fulldate = '{day}/{month}/{year}'.format(
            day=self.day[:2],
            month=self.day[2:4],
            year=self.day[4:]
            )
        xspf['playlist']['title'] = fulldate
        xspf['playlist']['trackList'] = {'track': []}
        other = {
            'extension': {
                'vlc:option': [],
                'vlc:item': [],
                '@application': 'http://www.videolan.org/vlc/playlist/0'
                }
            }
        now = time.time()
        infos = self.infos(now)
        for timestamp, path in infos.items():
            break  # on récupère le début de la programmation
        diff = now - timestamp
        steps = itertools.izip(
            *[
                itertools.chain(
                    itertools.islice(infos, i, None),
                    itertools.islice(infos, None, i)
                    )
                for i in range(2)
                ]
            )
        index = 0
        for start, stop in steps:
            uri = 'http://www.videolan.org/vlc/playlist/0'
            index += 1
            extension = {
                '@application': uri,
                'vlc:id': index,
                'vlc:option': []
            }
            filepath = infos[start].split(':')[0]
            duration = stop - start
            if index == 1:
                if diff < 0:
                    time.sleep(abs(diff))
                else:
                    extension['vlc:option'].append(
                        'start-time={0}'.format(diff)
                        )
            if stop <= start:
                stop = self.start_timestamp + 86340
            interval = stop - start
            if duration < interval:
                repeat = int(round(interval/duration)) - 1
                if repeat < 0:
                    return None
                elif repeat:
                        extension['vlc:option'].append(
                            'input-repeat={0}'.format(repeat)
                            )
            track = {
                'location': 'file:///{0}'.format(filepath),
                'duration': duration
                }
            track['extension'] = extension
            xspf['playlist']['trackList']['track'].append(track)
            other['extension']['vlc:item'].append({'@tid': str(index)})
        document = xmltodict.unparse(xspf, pretty=True)
        document = document.replace('></vlc:item>', '/>')
        extensions = xmltodict.unparse(other, pretty=True)
        extensions = extensions.replace(
            '<?xml version="1.0" encoding="utf-8"?>',
            ''
            )
        extensions = extensions.replace('\n', '\n\t')
        extensions = extensions.replace('></vlc:item>', '/>')
        result = document.split(
            '</playlist>')[0]+extensions+'\n</playlist>'
        return result

    def stream(self):
        """Diffusion via VLC"""
        xspf = self.xspf()
        xspf_file = os.path.join(
            CONFIG['APP']['playlists_folder'],
            self.redis_zset_key.replace(':', '_')
            )
        xspf_file += '.xspf'
        with open(xspf_file, 'w') as f:
            f.write(xspf)
        vlc(xspf_file, self.service_id)

# EOF
