#!/usr/bin/env python
# -*- coding: utf-8 -*-

from datetime import datetime
from email.header import Header
from email.mime.text import MIMEText
from email.utils import formatdate, make_msgid
from smtplib import SMTP

from ..consts import CONFIG, HOSTNAME, MEDIA, PROGRAMS, SERVICES
from .. import cluster, media, program, streaming
from . import app

import itertools
import time

import click


@click.group()
def cmd():
    pass


@cmd.command()
@click.option('--host', default='0.0.0.0')
@click.option('--port', default=5000)
def runserver(host, port):
    """Lancer le serveur Werkzeug pour l'API REST (dev. seulement)"""
    app.run(host=host, port=port, debug=True)


@cmd.command()
def ping():
    """Découverte du serveur de diffusion principale"""
    result = cluster.discover_master_node()
    if not result:
        click.echo("\nIl n'y a pas de master dans ce cluster")
    elif result is True:
        click.echo("\nJe suis le master")
    else:
        click.echo("\n{} est le master".format(result['hostname']))
    click.echo('\n')


@cmd.command()
@click.option('--hostname')
def enable(hostname):
    """Marque un serveur comme étant serveur de diffusion principal"""
    result = cluster.set_master_node(hostname)
    if not result:
        click.echo("\nJe suis déjà le master")
    elif result is True:
        click.echo("\nJe suis maintenant le master")
    else:
        click.echo("\n{} est maintenant le master".format(hostname))
    click.echo('\n')


@cmd.command()
@click.argument('filename')
@click.option('--image', help='Fichier image à incruster')
@click.option('--date', help="Date d'incrustation")
@click.option('--service-id', help="Numéro de chaine")
def merge(filename, image, date, service_id):
    """Fusion d'un fichier TS et d'un fichier image"""
    kwargs = {
        'filename': filename,
        'image': image,
        'day': date,
        'service_id': service_id
        }
    media.embed_image_in_tsfile(**kwargs)


@cmd.command()
@click.argument(
    'obj',
    type=click.Choice(['media', 'program']),
    metavar='media|program'
    )
@click.argument('name')
@click.option(
    '--service-id',
    default=85,
    help='Numéro de chaine'
    )
@click.option(
    '--output',
    default=1,
    help='Port de sortie sur la carte Dektec'
    )
def play(obj, name, service_id, output):
    """Diffusion via DtPlay d'un media ou d'un programme entier"""
    if obj == 'media':
        tsfile = media.Media(name)
        if not tsfile.exists:
            click.echo(
                "Désolé le fichier {} n'existe pas".format(tsfile.filepath)
                )
        else:
            streaming.dtplay(media.filepath, service_id, output)
    else:
        p = program.Program(name, service_id)
        p.play(output)


@cmd.command()
@click.argument(
    'obj',
    type=click.Choice(['media', 'program']),
    metavar='media|program'
    )
@click.argument(
    'name',
    metavar='MEDIA_NAME_OR_PROGRAM_DATE'
    )
def delete(obj, name):
    """Suppression d'un media ou d'un programme. \
    Le nom d'un programme est sa date de diffusion au format jjmmaaaa.
    """
    if obj == 'media':
        tsfile = media.Media(name)
        tsfile.delete()
        click.echo('\n{} supprimé'.format(tsfile.filepath))
    else:
        PROGRAMS.delete(name)
        click.echo('\nProgrammation {} supprimée'.format(name))
    click.echo('\n')


@cmd.command()
@click.argument(
    'obj',
    type=click.Choice(['media', 'program']),
    metavar='media|program'
    )
@click.argument(
    'name',
    metavar='MEDIA_NAME_OR_PROGRAM_DATE'
    )
@click.option(
    '--service-id',
    help='Numéro de chaine'
    )
@click.option(
    '--remote',
    help='Serveur distant avec lequel se synchroniser'
    )
def sync(obj, name, service_id, remote=None):
    """Synchronisation des media ou des programmes avec les pairs.
    La synchronisation des programmes se fait uniquement avec
    le master.
    """
    if obj == 'program':
        if not service_id:
            click.echo("\nIl faut préciser l'option 'service-id'")
        else:
            try:
                program.Program(name, service_id).sync(remote)
            except SystemError:
                click.echo(
                    "\nAucune synchronisation: pas de serveur master trouvé"
                    )
            else:
                click.echo("\nSynchronisation des programmes terminée")
    elif name == 'emediaplace':
        # récupère le fichier emediaplace du jour
        click.echo('\nRécupération du fichier emediaplace')
        fileheader = CONFIG['EMEDIAPLACE']['fileheader']
        today = time.strftime('%Y%m%d')
        filename = '{0}_{1}.ts'.format(fileheader, today)
        media.sync_file(filename)
    else:
        result = media.sync_file(name, remote=remote)
        if not result:
            click.echo("\nAucune synchronisation n'a été faite")
        else:
            click.echo("\nLes fichiers suivants ont été copiés:")
            for peer, movies in result.items():
                title = "depuis {}:".format(peer)
                click.echo("\n{}\n{}".format(title, '='*len(title)))
                click.echo("\n".join(movies))
    click.echo('\n')


@cmd.command()
@click.argument(
    'obj',
    type=click.Choice(['media', 'program']),
    metavar='media|program'
    )
@click.argument(
    'name',
    metavar='MEDIA_NAME_OR_PROGRAM_DATE'
    )
@click.option(
    '--service-id',
    help='Numéro de chaine'
    )
@click.option(
    '--coherence',
    is_flag=True,
    default=False,
    help='Vérification de la cohérence du programme'
    )
@click.option(
    '--send',
    default=None,
    help='Envoi du résultat à une adresse mail'
    )
def check(obj, name, service_id, coherence, send):
    """Commande qui permet de vérifier un objet 'media' ou 'program'"""
    if obj == 'program':
        if not service_id:
            service_ids = [
                i.split(':')[1] for i in CONFIG.sections()
                if i.startswith('SERVICE:')
                ]
        else:
            service_ids = [service_id]
        services = {
            service_id: CONFIG['SERVICE:{}'.format(service_id)]['name']
            for service_id in service_ids
            }
        msg = ''
        for service_id, service_name in services.items():
            p = program.Program(name, service_id)
            if not coherence:
                infos = p.infos()
                click.echo('\n{} ({})'.format(service_name, name))
                click.echo('='*(len(service_name)+len(name)+3))
                timestamps = sorted(infos)
                for timestamp in timestamps:
                    path = infos[timestamp]
                    info = (
                        "{timestamp}: {filepath}".format(
                            timestamp=datetime.fromtimestamp(
                                timestamp).strftime('%d/%m/%Y %H:%M:%S'),
                            filepath=path.split(':')[0]
                            )
                        )
                    click.echo(info)
            else:
                result = p.check()
                statuses = {
                    True: 'OK',
                    False: 'NOK',
                    None: 'Impossible à vérifier'
                    }
                msg += '{0}\n{1}\n\n'.format(
                    service_name,
                    '='*len(service_name)
                    )
                msg += '{0}:\n{1} '.format(name, '-'*len(name))
                msg += "\nStatus de la base redis: {}\n".format(
                    statuses[result['redis']])
                if type(result['program']) is list:
                    bar = '\nLes fichiers suivants manquent:\n{}'.format(
                        '\n'.join(list(set(result['program'])))
                        )
                else:
                    bar = statuses[result['program']]
                if result.get('overflow'):
                    bar += '\nErreur: la programmation fini le {}'.format(
                        datetime.fromtimestamp(
                            result['overflow']).strftime(
                            '%d-%m-%Y à %H:%M:%S')
                        )
                msg += "\nProgrammation: {}\n".format(bar)
                msg += "\nEPG sur http://epg.blueline.mg: {}\n".format(
                    statuses[result['epg']])
                msg += "\nCohérence EPG & Programmation: {}\n\n".format(
                    statuses[result['coherence']])
        if not send:
            click.echo('\n')
            click.echo(msg)
        elif send and msg:
            # send via email
            subject = (
                "[{0}.malagasy.com] dvbbox check program {1}"
                ).format(HOSTNAME, name)
            if send == 'all':
                dst = [
                    "monitoring@gulfsat.mg",
                    "admin-tv@gulfsat.mg",
                    "ctv@gulfsat.mg"
                    ]
            else:
                dst = [send]
                orig = "dvb@{0}.malagasy.com".format(HOSTNAME)
                message_to_send = MIMEText(msg, 'plain', 'utf-8')
                message_to_send.set_charset('utf-8')
                message_to_send['Subject'] = Header(subject, 'utf-8')
                message_to_send['From'] = orig
                message_to_send['To'] = ", ".join(dst)
                message_to_send['Date'] = formatdate(localtime=True)
                message_to_send['Message-ID'] = make_msgid()
                mailer = SMTP('localhost')
                mailer.sendmail(orig, dst, message_to_send.as_string())
            click.echo("\nCourriel envoyé à {}".format(', '.join(dst)))
        click.echo('\n')
    else:
        tsfile = media.Media(name)
        if not tsfile.exists:
            click.echo(
                "Désolé le fichier {} n'existe pas".format(tsfile.filepath)
                )
        elif not coherence:
            m, s = divmod(tsfile.duration, 60)
            h, m = divmod(m, 60)
            duration = "%d:%02d:%02d" % (h, m, s)
            click.echo("\n{} (Durée: {})\n".format(tsfile.filepath, duration))
            infos = tsfile.schedule(service_id=service_id)
            if not infos:
                click.echo("Aucune programmation prévue pour ce fichier")
            program_keys = sorted(infos, key=lambda x: int(x.split(':')[1]))
            grouped_program_keys = itertools.groupby(
                program_keys,
                key=lambda x: int(x.split(':')[1])
                )
            for service_id, program_keys in grouped_program_keys:
                service_name = CONFIG['SERVICE:{}'.format(service_id)]['name']
                timestamps = []
                for program_key in program_keys:
                    timestamps += infos[program_key].keys()
                timestamps = sorted(timestamps)
                click.echo(service_name)
                click.echo('='*len(service_name))
                for timestamp in timestamps:
                    click.echo(
                        datetime.fromtimestamp(timestamp).strftime(
                            '%d/%m/%Y %H:%M:%S')
                        )
            click.echo('\n')
        else:
            # on devrait pouvoir lancer un test sur le fichier TS
            # pour vérifier son intégrité
            click.echo("Vérification de cohérence non implémentée")


@cmd.command()
@click.argument(
    'obj',
    type=click.Choice(['media', 'program']),
    metavar='media|program'
    )
@click.argument(
    'name',
    metavar='MEDIA_NAME_OR_PROGRAM_DATE'
    )
@click.option(
    '--service-id',
    help='Numéro de chaine'
    )
def update(obj, name, service_id):
    """Mise à jour des informations de media ou programme"""
    if obj == 'media':
        if name == 'all':  # on met tout à jour
            with click.progressbar(media.Media.all()) as tsfiles:
                for tsfile in tsfiles:
                    MEDIA.set(tsfile.filename, tsfile.duration)
        else:
            media.Media(name).__update__()
    else:
        if not service_id:
            programs = [
                program.Program(name, service_id)
                for service_id in SERVICES
                ]
        else:
            programs = [program.Program(name, service_id)]
        for p in programs:
            p.update()


if __name__ == '__main__':
    cmd()
