# -*- coding: utf-8 -*-

from .consts import CONFIG
from .consts import MEDIA
from .consts import MEDIA_SYNOPSIS_DB
from .consts import PROGRAMS
from . import program
from . import cluster

import ftplib
import os
import re
import shlex
import shutil
import subprocess
import time
import threading

import records


__all__ = [
    'Media',
    'embed_image_in_tsfile',
    'sync_file',
    ]


class Media(object):
    """Représentation d'un fichier TS"""

    def __init__(self, filename):
        self.filename = filename
        if not self.filename.endswith('.ts'):
            self.filename += '.ts'
        self.filepath = '{root}/{filename}'.format(
            root=CONFIG['APP']['media_folder'],
            filename=self.filename
            )

    def __repr__(self):
        return '<Media {}>'.format(self.filepath)

    def __update__(self):
        MEDIA.delete(self.filename)
        if self.exists:
            MEDIA.set(self.filename, self.duration)

    @staticmethod
    def __all__():
        """Retourne la liste complète des fichiers media existants.
        Juste les noms des fichiers"""
        return [
            i for i in os.listdir(CONFIG['APP']['media_folder'])
            if i.endswith('.ts')
            ]

    @staticmethod
    def all():
        """Retourne la liste complète des fichiers media existants"""
        return [Media(i) for i in Media.__all__()]

    @staticmethod
    def update():
        """Met à jour le cache de fichiers existant sur le disque"""
        MEDIA.flushdb()
        for media in Media.all():
            MEDIA.set(media.filename, media.duration)

    @property
    def exists(self):
        """Renvoie un booléen pour savoir si le fichier existe ou pas"""
        return self.filename in os.listdir(CONFIG['APP']['media_folder'])

    def details(self):
        """Titre et description du film (le synopsis)"""
        filename = self.filename.replace('.ts', '')
        with records.Database(MEDIA_SYNOPSIS_DB) as db:
            sql = (
                "SELECT name, description "
                "FROM media WHERE filename=:filename"
                )
            rows = db.query(sql, filename=filename)
            info = rows.first()
            return info

    @property
    def duration(self):
        """Calcul de la durée en secondes"""
        cmd = "avprobe -show_format {}".format(self.filepath)
        p = subprocess.Popen(
            shlex.split(cmd), stderr=subprocess.PIPE, stdout=subprocess.PIPE
            )
        result = p.communicate()[0]
        duration_regex = re.compile(b'.*\nduration=(.*)\n.*')
        try:
            value = duration_regex.search(result).groups()[0].strip()
            value = float(value)
        except Exception as exc:  # normalement c'est un IndexError
            value = float(3.14)  # impossible de calculer la durée
        finally:
            return value

    def copy_from(self, remote_host):
        """Copie du fichier media depuis un serveur distant"""
        remote_path = '{remote_host}:{source}'.format(
            remote_host=remote_host,
            source=self.filepath
            )
        if cluster.Server().usage('/opt') < 0.9:
            cmd = 'rsync --progress -z {0} {1}'.format(
                remote_path,
                self.filepath
                )
            subprocess.call(shlex.split(cmd))

    def schedule(self, day=None, service_id=None):
        """Retourne les 'timestamps' auxquels le fichier
        doit être diffusé.
        Le résultat est un dictionnaire dont les clés sont
        une chaine de caractères sous la forme "jour:chaine",
        et les valeurs un dicitonnaire dont les clés sont
        les heures de diffusion et les valeurs les index dans la base
        """
        if not day and not service_id:
            keys = PROGRAMS.keys()
            keys = [key.decode('utf-8') for key in keys]
        elif day and not service_id:
            keys = PROGRAMS.keys('{day}:*'.format(day=day))
            keys = [key.decode('utf-8') for key in keys]
        elif not day and service_id:
            keys = PROGRAMS.keys(
                '*:{service_id}'.format(service_id=service_id)
                )
            keys = [key.decode('utf-8') for key in keys]
        else:
            keys = [
                '{day}:{service_id}'.format(day=day, service_id=service_id)
                ]
        result = {}
        for key in keys:
            p = program.Program(*key.split(':'))
            occurences = p.get_filename_schedule(self.filename)
            if occurences:
                result[key] = occurences
        return result

    def embed_image(self, image='panneau_blueline.png', **kwargs):
        """Créé une version modifiée du fichier media
        en y incrustant une image.
        Le script utilisé pour ça est /usr/local/bin/ffmpeg_embed_image.
        La version modifiée est dans CONFIG['APP']['modified_media_folder']
        """
        service_id = kwargs.get('service_id', 1)
        day = kwargs.get('day', time.strftime('%d%m%Y'))
        if self.filename.startswith('revue_emediaplace'):
            day = self.filename.split('revue_emediaplace_')[1].split('_')[0]
            day = day[6:] + day[4:6] + day[:4]
            service_id = CONFIG['EMEDIAPLACE']['service']
        image_filepath = os.path.join(
            CONFIG['APP']['images_folder'],
            image
            )
        section = 'SERVICE:{}'.format(service_id)
        pid_video = CONFIG[section]['pid_video']
        pid_audio = CONFIG[section]['pid_audio']
        new_folder = '{root}/{day}/{service_id}'.format(
            root=CONFIG['APP']['modified_media_folder'],
            day=day,
            service_id=service_id
            )
        subprocess.call(
            shlex.split(
                'mkdir -p {}'.format(new_folder)
                )
            )
        new_filepath = '{new_folder}/{filename}'.format(
            new_folder=new_folder,
            filename=self.filename
            )
        if self.filename.startswith('revue_emediaplace'):
            # fondre l'image dans le fichier et retirer la bande son
            x264 = (
                'vbv-maxrate=2000:'
                'bitrate=2000:'
                'vbv-bufsize=80:'
                'tff=1:'
                'weightp=2:'
                'fps:25'
                )
            cmd = (
                'ffmpeg -threads auto -ss 3 '
                '-i {filepath} '
                '-loop 1 -i {image} '
                '-c:v libx264 '
                '-x264-params {x264} '
                '-s 720x576 '
                '-aspect 16:9 '
                '-an '
                '-y /tmp/v_revue_bluepano.ts'
                ).format(
                    filepath=self.filepath,
                    x264=x264,
                    image=image_filepath
                    )
            subprocess.call(shlex.split(cmd))
            # ajouter la bande son
            audio = os.path.join(
                CONFIG['APP']['audios_folder'],
                'audio_emediaplace.aac'
                )
            cmd = (
                'ffmpeg -threads auto '
                '-i /tmp/v_revue_bluepano.ts '
                '-i {audio} '
                '-map 0:v -map 1:a '
                '-c:v copy -c:a copy '
                '-y /tmp/av_revue_bluepano.ts'
                ).format(audio=audio)
            subprocess.call(shlex.split(cmd))
            # remixer le tout
            concat = 'concat:'+'|'.join(
                ['/tmp/av_revue_bluepano.ts' for i in range(5)]
                )
            cmd = (
                'ffmpeg -threads auto -strict experimental '
                '-i "{concat}" '
                '-c:v libx264 '
                '-x264-params {x264} '
                '-s 720x576 '
                '-aspect 16:9 '
                '-c:a aac -ac 2 -b:a 128k '
                '-mpegts_transport_stream_id 1 '
                '-mpegts_service_id {service_id} '
                '-streamid 0:{pid_video} '
                '-streamid 1:{pid_audio} '
                '-metadata service_provider="Blueline TV" '
                '-metadata service_name="Emediaplace" '
                '-f mpegts '
                '-muxrate 2000k '
                '-shortest '
                '-y {output}'
                ).format(
                    concat=concat,
                    x264=x264,
                    service_id=service_id,
                    pid_video=pid_video,
                    pid_audio=pid_audio,
                    output=new_filepath
                    )
            subprocess.call(shlex.split(cmd))
            subprocess.call(shlex.split('rm /tmp/v_revue_bluepano.ts'))
            subprocess.call(shlex.split('rm /tmp/av_revue_bluepano.ts'))
            return
        else:
            schedule = self.schedule(day, service_id)
            key = '{}:{}'.format(day, service_id)
            if schedule:
                moment_0_start = self.duration - 900
                moment_0_stop = self.duration - 890
                moment_1_start = self.duration - 600
                moment_1_stop = self.duration - 590
                moment_2_start = self.duration - 300
                moment_2_stop = self.duration - 290
                filter_cmd = (
                    '[1:v]fade=in:st={m0_start}:d=0.5,'
                    'fade=out:st={m0_stop}:d=0.5[w1];'
                    '[0:v][w1]overlay=353:434:shortest=1[video1];'
                    '[2:v]fade=in:st={m1_start}:d=0.5,'
                    'fade=out:st={m1_stop}:d=0.5[w2];'
                    '[video1][w2]overlay=353:434:shortest=1[video1];'
                    '[3:v]fade=in:st={m2_start}:d=0.5,'
                    'fade=out:st={m2_stop}:d=0.5[w3];'
                    '[video1][w3]overlay=353:434:shortest=1'
                    ).format(
                        m0_start=moment_0_start,
                        m0_stop=moment_0_stop,
                        m1_start=moment_1_start,
                        m1_stop=moment_1_stop,
                        m2_start=moment_2_start,
                        m2_stop=moment_2_stop
                        )
                x264opts = (
                    'bitrate=2500:'
                    'vbv-maxrate=2500:'
                    'vbv-bufsize=166:'
                    'tff=1:'
                    'weightp=2:'
                    'fps=25'
                    )
                cmd = (
                    'ffmpeg -threads auto -i {filepath} '
                    '-loop 1 -i {image} -loop 1 -i {image} -loop 1 -i {image} '
                    '-filter_complex "{filter_cmd}" '
                    '-c:v libx264 -profile:v high -level 3.0 '
                    '-x264opts {x264opts} '
                    '-s 720x576 '
                    '-aspect 20:11 '
                    '-c:a mp2 '
                    '-ac 2 '
                    '-b:a 128k '
                    '-mpegts_transport_stream_id 1200 '
                    '-mpegts_service_id {service_id} '
                    '-mpegts_start_pid {start_pid} '
                    '-metadata service_provider="CTV" '
                    '-metadata service_name="CTV" '
                    '-f mpegts '
                    '-muxrate 2500k '
                    '-y {output}'
                    ).format(
                        filepath=self.filepath,
                        image=image_filepath,
                        filter_cmd=filter_cmd,
                        x264opts=x264opts,
                        service_id=service_id,
                        start_pid=pid_video,
                        output=new_filepath
                        )
                subprocess.call(shlex.split(cmd))
                for timestamp, index in schedule[key].items():
                    PROGRAMS.zadd(
                        key,
                        '{}:{}'.format(new_filepath, index),
                        timestamp
                        )
                    PROGRAMS.zrem(
                        key,
                        '{}:{}'.format(self.filepath, index),
                        )
                return

    def rename(self, new_name):
        """Renommer le fichier"""
        schedule = self.schedule()
        old_path = self.filepath
        new_path = '{root}/{new_name}'.format(
            root=CONFIG['APP']['media_folder'],
            new_name=new_name
            )
        shutil.move(old_path, new_path)
        for redis_zset_key, infos in schedule.items():
            for timestamp, index in infos.items():
                old_entry = '{}:{}'.format(
                    old_path,
                    index
                    )
                new_entry = '{}:{}'.format(
                    new_path,
                    index
                    )
                PROGRAMS.zadd(redis_zset_key, new_entry, timestamp)
                PROGRAMS.zrem(redis_zset_key, old_entry)
        self.__init__(new_name)

    def delete(self):
        """Supprimer le fichier et réajuster
        la programmation en conséquence
        """
        duration = self.duration
        schedule = self.schedule()
        now = time.time()
        tmstps = []
        if schedule:
            # os.remove(self.filepath)                                                                                                                                    
            for key, tmstp in schedule.items():
                t = [j for j in tmstp if j >= now]
                tmstps += t
            tmstps = sorted(list(set(tmstps)))

            if not tmstp:
                os.remove(self.filepath)
                for redis_zset_key, infos in schedule.items():
                    timestamps = sorted(infos)
                    p = program.Program(
                        *redis_zset_key.split(':')
                         ).infos(timestamps[0])
                    factor = 0
                    for timestamp, path in p.items():
                        # à chaque fois qu'on rencontre une occurence du fichier                                                                                          
                        # on décrémente le facteur de multiplication de 1                                                                                                 
                        # pour que l'heure de début du programme                                                                                                          
                        # venant après ladite occurence                                                                                                                   
                        # soit réduite de la bonne valeur                                                                                                                 
                        if timestamp in timestamps:
                            factor -= 1
                            PROGRAMS.zrem(redis_zset_key, path)
                        else:
                            PROGRAMS.zincrby(
                                redis_zset_key,
                                path,
                                (factor * duration)
                                )
            else:
                raise ValueError('Cannot delete for it is scheduled')
        return


def embed_image_in_tsfile(**kwargs):
    """Cette fonction permet d'incruster un fichier
    image dans un fichier TS.
    Pour des raisons de syntaxe (80 col).
    la signature de la fonction est volontairement escamotée
    """
    filename = kwargs.get('filename')
    image = kwargs.get('image')
    service_id = kwargs.get('service_id', 1)
    day = kwargs.get('day', time.strftime('%d%m%Y'))
    if filename.startswith('revue_emediaplace'):
        Media(filename).embed_image()
    elif day and service_id:
        if image and filename:
            tsfile = Media(filename)
            tsfile.embed_image(image=image, day=day, service_id=service_id)
        else:
            # on incruste suivant le mapping
            mapping = os.path.join(
                CONFIG['APP']['modified_media_folder'],
                day,
                service_id,
                'mapping.txt'
                )
            with open(mapping) as f:
                content = f.readlines()
            mapping = {
                line.split(';')[0]: line.split(';')[1].replace('\n', '')
                for line in content
                }
            if filename:
                image = mapping[filename]
                tsfile = Media(filename)
                tsfile.embed_image(image=image, day=day, service_id=service_id)
            else:
                class MyWorker(threading.Thread):

                    def __init__(self, filename, image, day, service_id):
                        threading.Thread.__init__(self)
                        self.image = image
                        self.day = day,
                        self.service_id = service_id
                        self.tsfile = cluster.Media(filename)

                    def run(self):
                        self.tsfile.embed_image(
                            image=self.image,
                            day=self.day,
                            service_id=self.service_id
                            )
                threads = [
                    MyWorker(filename, image, date=day, service_id=service_id)
                    for filename, image in mapping.items()
                    ]
                for thread in threads:
                    thread.start()


def sync_file(name=None, remote=None):
    """Cette fonction synchronise les fichiers depuis les serveurs pairs"""
    if name.startswith('revue_emediaplace'):
        tsfile = Media(name.replace('.ts', '_ctv5.ts'))
        # récupère le fichier emediaplace
        ftp_server = CONFIG['EMEDIAPLACE']['ftp_server']
        ftp_user = CONFIG['EMEDIAPLACE']['ftp_user']
        ftp_password = CONFIG['EMEDIAPLACE']['ftp_password']
        ftp = ftplib.FTP(ftp_server)
        ftp.login(ftp_user, ftp_password)
        with open(tsfile.filepath, 'wb') as f:
            ftp.retrbinary('RETR {0}'.format(name), f.write)
        ftp.quit()
        return
    else:
        # va mettre à jour le répertoire local avec le contenu des les pairs
        server = cluster.Server()
        peers = CONFIG['PEERS']
        result = {}
        if remote:
            peers = [remote]
        for peer in peers:
            peer_server = cluster.Server(peer)
            peer_movies = peer_server.media()
            local_movies = server.media()
            if not name:
                movies_to_copy = [
                    peer_movie for peer_movie in peer_movies
                    if peer_movie not in local_movies
                    ]
                result[peer] = movies_to_copy
                for movie_to_copy in movies_to_copy:
                    tsfile = Media(movie_to_copy)
                    tsfile.copy_from(peer_server.hostname)
            elif name in peer_movies:
                result[peer] = name
                movie_to_copy = Media(name)
                movie_to_copy.copy_from(peer_server.hostname)
        return result
# EOF
