# -*- coding: utf-8 -*-

from datetime import datetime

from .consts import CONFIG, PROGRAMS
from . import media

import json
import os
import shlex
import subprocess
import time

import pika
import redis
import yaml


class Listing(object):
    """Représentation d'un listing.
    Un listing est un fichier YAML dont les sections sont des jours
    et les valeurs une succession de fichiers media à diffuser.
    """
    def __init__(self, filename):
        """Le format du nom du fichier de listing doit être
        service-id_jour-début_jour-fin
        """
        try:
            service_id, day_start, day_stop = filename.split('_')
        except ValueError:
            raise ValueError(
                (
                    "Le listing {} ne respecte pas "
                    "le format service-id_jour-début_jour-fin"
                    )
                )
        else:
            self.service_id = service_id
            self.filepath = os.path.join(
                CONFIG['APP']['uploads_folder'],
                filename
                )
            self.filename = filename
            # on va parcourir le listing pour avoir
            # la liste complète des jours ainsi que celle des fichiers media
            with open(self.filepath) as f:
                listing = yaml.load(f)
            self.content = listing
            self.days = sorted([i for i in self.content])
            self.filenames = {}
            filenames = list(
                set(
                    [
                        filename for filenames in self.content.values()
                        for filename in filenames
                        ]
                    )
                )
            filenames = [i.split(';')[0] for i in filenames]
            for filename in filenames:
                m = media.Media(filename)
                duration = m.duration
                self.filenames[filename] = [m.filepath, duration]

    def parse_one_day(self, day, apply=False):
        """Méthode qui prend en paramètres
            - une date (str) au format jj/mm/aaaa
        et retourne la playlist avec les heures de début de diffusion
        de chaque fichier.
        Si on le précise, le résultat du parsing est appliqué
        (i.e enregistré dans la base REDIS)
        """
        result = {}
        start_time = CONFIG['SERVICE:{}'.format(self.service_id)]['start']
        start = time.mktime(
            time.strptime(
                '{day} {start_time}'.format(
                    day=day,
                    start_time=start_time
                    ),
                '%d/%m/%Y %H%M%S'
                )
            )
        for filename in self.content[day]:
            if ';' in filename:
                # si un fichier est marqué pour modification
                # par incrustation de panneau
                path = os.path.join(
                    CONFIG['APP']['modified_media_folder'],
                    day.replace('/', ''),
                    self.service_id
                    )
                subprocess.call(
                    shlex.split(
                        'mkdir -p {}'.format(path)
                        )
                    )
                filename, image = filename.split(';')
                with open(path+'/mapping.txt', 'a') as f:
                    f.write(filename+'.ts;'+image)
            filepath, duration = self.filenames[filename]
            result[start] = {
                'filepath': filepath,
                'duration': duration
                }
            start += duration
        if apply:  # on applique le parsing
            day = day.replace('/', '')
            zset_key = '{0}:{1}'.format(day, self.service_id)
            PROGRAMS.delete(zset_key)
            timestamps = sorted(result)
            index = 0
            for timestamp in timestamps:
                infos = result[timestamp]
                filepath = infos['filepath']
                PROGRAMS.zadd(
                    zset_key, filepath+':'+str(index), float(timestamp)
                    )
                index += 1
        return {day: result}

    def parse(self):
        """Analyse du fichier"""
        data = {}
        from multiprocessing import Pool
        with Pool(4) as p:
            result = p.map(self.parse_one_day, self.content.keys())
        for i in result:
            data.update(i)
        return data

    def apply(self, data):
        """Application du listing"""
        for day, program in data.items():
            day = day.replace('/', '')
            zset_key = '{0}:{1}'.format(day, self.service_id)
            PROGRAMS.delete(zset_key)
            timestamps = sorted(program)
            index = 0
            for timestamp in timestamps:
                infos = program[timestamp]
                filepath = infos['filepath']
                PROGRAMS.zadd(
                    zset_key, filepath+':'+str(index), float(timestamp)
                    )
                index += 1


def parse_from_rabbitmq_queue(service_id):

    def callback(ch, method, properties, body):
        try:
            filename = json.loads(body.decode('utf-8'))
        except Exception as exc:
            ch.basic_ack(delivery_tag=method.delivery_tag)
        else:
            listingfile = Listing(filename)
            data = listingfile.parse()
            message = {
                'programs': [
                    {
                        'date': date,
                        'schedule': [
                            {
                                'date': datetime.fromtimestamp(
                                    timestamp
                                    ).strftime(
                                        '%d%m%Y'
                                        ),
                                'hour': datetime.fromtimestamp(
                                    timestamp
                                    ).strftime(
                                        '%H:%M:%S'
                                        ),
                                'timestamp': timestamp,
                                'filepath': infos['filepath']
                                }
                            for timestamp, infos in program.items()
                            ]
                        }
                    for date, program in data.items()
                    ]
                }
            r = redis.Redis(db=2)
            r.set(listingfile.filename, message, exc=3600)
    connection = pika.BlockingConnection(
        pika.ConnectionParameters(
            host='localhost',
            virtual_host='/dvbbox'
            )
        )
    channel = connection.channel()
    channel.basic_qos(prefetch_count=1)
    channel.basic_consume(callback, queue=service_id)
    channel.start_consuming()

# EOF
