#!/usr/bin/env python
# encoding: utf-8


from functools import partial
from multiprocessing.pool import Pool

from .consts import CONFIG
from .consts import REST_CLIENT
from . import core

__all__ = [
    'Product'
    ]


class Product(object):
    """Toute politique de bon usage est centrée autour d'un produit.
    Un produit est identifié par son nom, et ses politiques de bon usage"""

    def __init__(self, name):
        self.name = name
        self.policies = {}
        policies_names = CONFIG['POLICIES']
        policy_names = [
            policy_name for policy_name in policies_names
            if policy_name.startswith(self.name)
            ]
        for policy_name in policy_names:
            data_limit, interval = policies_names[policy_name].split(',', 1)
            self.policies[policy_name] = {
                'data_limit': int(data_limit),
                'interval': interval
                }

    def __repr__(self):
        return '<Product {name}>'.format(name=self.name)

    @property
    def customers(self):
        """Récupération de la liste des clients possédant ce produit"""
        from .customer import Customer
        customers = [
            c for c in Customer.all()
            if c.product.name == self.name
            ]
        return customers

    @staticmethod
    def all():
        policy_names = [i for i in CONFIG['POLICIES']]
        products = set(['_'.join(i.split('_')[:-1]) for i in policy_names])
        return [Product(name) for name in products]

    def apply_policies(self, for_real=False):
        """Application des politiques de bon usage à tous les clients
        abonnés à ce produit."""

        f = partial(core.apply_product_policies, for_real)
        with Pool(4) as p:
            response = p.map(f, self.customers)
        customers = {i[0].name: i[1] for i in response if i[1]}
        policies = []
        for policy_name, policy in self.policies.items():
            policy_infos = {
                'data_limit': policy['data_limit'],
                'name': policy_name,
                'interval': policy['interval'],
                'customers': {}
                }
            policy_customers = policy_infos['customers']
            for customer_id, data in customers.items():
                for info in data:
                    if info['name'] == policy_name:
                        policy_customers[customer_id] = {
                            'total': info['total'],
                            'octets_in': info['octets_in'],
                            'octets_out': info['octets_out'],
                            'details': info['details']
                            }
                        if 'fup_in' in info:
                            policy_customers[customer_id]['fup_in'] = True
                        elif 'fup_out' in info:
                            policy_customers[customer_id]['fup_out'] = True
            policies.append(policy_infos)
        return policies

    def update_customers(self):
        """Récupération de la nouvelle liste de clients depuis aiguillier"""
        response = REST_CLIENT.get(
            '/fup',
            auth=(
                CONFIG['AIGUILLIER']['username'],
                CONFIG['AIGUILLIER']['password']
                ),
            params={'product': self.name}
            )
        customers = response['content'].get('customers', [])
        if customers:
            existing_customers_names = [
                customer.name for customer in self.customers
                ]
            new_customers = filter(
                lambda x: x['name'] not in existing_customers_names,
                customers
                )
            my_pool = Pool(4)
            my_pool.map(core.record_customer, new_customers)

# EOF
