# -*- coding: utf-8 -*-

from datetime import datetime, timedelta
from itertools import groupby

from .consts import CONFIG
from .consts import FUPDB_CREDENTIALS
from .consts import LOGS
from .consts import REDIS_DB
from .consts import REST_CLIENT
from . import core
from . import product

import records


__all__ = [
    'Customer',
    ]


class Customer(object):
    """Un client internet est identifié par un nom, un refnum et un produit"""

    def __init__(self, name):
        self.name = name.lower()
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = "SELECT refnum, product FROM customer WHERE name=:name"
            rows = db.query(sql, name=self.name)
            row = rows.first()
            if not row:
                raise ValueError(
                    "Le client {name} n'existe pas".format(name=name)
                    )
            else:
                self.product = product.Product(row.product)
                self.refnum = row.refnum

    def __repr__(self):
        return '<Customer {name}>'.format(name=self.name)

    @staticmethod
    def all():
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = "SELECT name FROM customer"
            rows = db.query(sql)
            customers = [Customer(row.name) for row in rows]
        return customers

    @property
    def fup_status(self):
        """Retourne le status FUP du client"""
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = "SELECT * FROM fup WHERE customer=:name"
            rows = db.query(sql, name=self.name)
            result = {}
            for row in rows:
                if row.policy not in result:
                    result[row.policy] = []
                result[row.policy].append(
                    {
                        'datetime': row.datetime,
                        'status': row.status
                        }
                    )
        return result

    def set_refnum(self, refnum):
        """Changement du refnum"""
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = (
                "UPDATE customer SET refnum=:refnum "
                "WHERE name=:name"
                )
            db.query(
                sql,
                refnum=refnum,
                name=self.name
                )
        log_msg = (
            "Changement de refnum de {}: {} => {}"
            ).format(
                self.name,
                self.refnum,
                refnum
                )
        LOGS.logger.info(log_msg)
        self.refnum = refnum

    def set_product(self, product_name):
        """Changement du produit"""
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = (
                "UPDATE customer SET product=:product "
                "WHERE name=:name"
                )
            db.query(
                sql,
                product=product_name,
                name=self.name
                )
        log_msg = (
            "Changement de produit de {}: {} => {}"
            ).format(
                self.name,
                self.product.name,
                product_name
                )
        LOGS.logger.info(log_msg)
        self.product = product.Product(product_name)

    def defup_from_aiguillier(self):
        """défuper un client qui est fupé"""
        result = {}
        customer_status = self.fup_status
        for policy_name in self.product.policies:
            status = customer_status.get(policy_name)
            if status:
                status = sorted(
                    status,
                    key=lambda x: x['datetime'],
                    reverse=True
                    )
                if status[0]['status']:
                    # si le client était fup, il faut le défup
                    REST_CLIENT.delete(
                        '/fup',
                        auth=(
                            CONFIG['AIGUILLIER']['username'],
                            CONFIG['AIGUILLIER']['password']
                            ),
                        params={'customer': self.name}
                        )
                    url = core.database_url(**FUPDB_CREDENTIALS)
                    with records.Database(url) as db:
                        sql = (
                            "INSERT INTO fup "
                            "(customer, product, policy, datetime, status) "
                            "VALUES "
                            "(:name, :product, :policy, :date_time, :status)"
                            )
                        db.query(
                            sql,
                            name=self.name,
                            product=self.product.name,
                            policy=policy_name,
                            date_time=datetime.now(),
                            status=False
                            )
                    result = {
                        'status': '200',
                        'message': 'défup reussit'
                        }
                else:
                    result = {
                            'status': '201',
                            'message': 'deja fupé'
                            }
            else:
                result = {
                        'status': '204',
                        'message': 'produit non fupé'
                        }

            LOGS.logger.info("Defuper un client {}-{}".format(
                self.name,
                policy_name
                ))
        return result

    def get_cdr_from_aiguillier(self, last_date):
        """Récupération des CDRs des 30 derniers jours
        depuis l'API d'Aiguillier pour les clients business"""
        url = core.database_url(**FUPDB_CREDENTIALS)
        LOGS.logger.warn('last date {} de {}'.format(last_date, self.name))
        response = REST_CLIENT.get(
            '/fup',
            auth=(
                CONFIG['AIGUILLIER']['username'],
                CONFIG['AIGUILLIER']['password']
                ),
            params={'customer': self.name, 'fromdate': last_date}
            )
        infos = response['content']
        # nill_condition = (
        #    'refnum' not in infos or
        #    infos['product'] not in [i.name for i in product.Product.all()]
        #    )
        # if nill_condition:
        # le client n'existe plus
        # ou son produit a changé et il n'est plus concerné
        # par le fup
        #    customer_status = self.fup_status
        #    for policy_name in self.product.policies:
        #        status = customer_status.get(policy_name)
        #        if status:
        #            status = sorted(
        #                status,
        #                key=lambda x: x['datetime'],
        #                reverse=True
        #                )
        #            if status[0]['status']:
        #                # si le client était fup, il faut le défup
        #                REST_CLIENT.delete(
        #                    '/fup',
        #                    auth=(
        #                        CONFIG['AIGUILLIER']['username'],
        #                        CONFIG['AIGUILLIER']['password']
        #                        ),
        #                    params={'customer': self.name}
        #                    )
        #    url = core.database_url(**FUPDB_CREDENTIALS)
        #    with records.Database(url) as db:
        #        sql = "DELETE FROM customer WHERE name=:name"
        #        db.query(sql, name=self.name)
        #        sql = "DELETE FROM cdr WHERE customer=:name"
        #        db.query(sql, name=self.name)
        #        sql = "DELETE FROM fup WHERE customer=:name"
        #        db.query(sql, name=self.name)
        #    LOGS.logger.info("Suppression du client {}".format(self.name))
        #    return None
        # else:
        cdrs = sorted(
            infos.get('cdr', []),
            key=lambda x: datetime.strptime(
                x['sessiontime'],
                '%Y-%m-%dT%H:%M:%S'
                )
            )
        cdrs = groupby(
            cdrs,
            key=lambda x: datetime.strptime(
                x['sessiontime'].split('T')[0],
                '%Y-%m-%d'
                )
            )
        result = {}
        params = []
        for date, infos in cdrs:
            LOGS.logger.info(
                "{} {} obtenu depuis Aiguillier".format(
                    self.name,
                    date.strftime('%d%m%Y')
                    )
                )
            result[date.strftime('%d%m%Y')] = []
            for info in infos:
                date_time = datetime.strptime(
                    info['sessiontime'],
                    '%Y-%m-%dT%H:%M:%S'
                    )
                params.append(
                    {
                        'name': self.name,
                        'product': self.product.name,
                        'date_time': date_time,
                        'source': info['source'],
                        'octetsin': info.get('octetsin', 0),
                        'octetsout': info.get('octetsout', 0)
                        }
                    )
                time = info['sessiontime'].split('T')[1].replace(':', '')
                values = {
                    'octets_in': info['octetsin'],
                    'octets_out': info['octetsout']
                    }
                values['time'] = datetime.strptime(time, '%H%M%S')
                values['source'] = info['source']
                values['date'] = date
                result[date.strftime('%d%m%Y')].append(values)
            with records.Database(url) as db:
                sql = (
                    "DELETE FROM cdr WHERE customer=:name "
                    "AND datetime>=:start AND datetime<=:stop"
                    )
                db.query(
                    sql,
                    name=self.name,
                    start=date.strftime('%Y-%m-%d 00:00:00'),
                    stop=date.strftime('%Y-%m-%d 23:59:59')
                    )
        if params:
            with records.Database(url) as db:
                sql = (
                    "INSERT INTO cdr "
                    "(customer, datetime, source, "
                    "octets_in, octets_out, product) "
                    "VALUES "
                    "(:name, :date_time, :source, "
                    ":octetsin, :octetsout, :product)"
                    )
                db.bulk_query(sql, *params)
        return result

    def get_cdr(self, start=None, stop=None):
        """Récupération des données de conso d'un client business.
        Les dates sont des objets datetime.datetime
        """
        if not start:
            start = (datetime.now() - timedelta(30))
        if not stop:
            stop = (datetime.now() - timedelta(1))
        if stop < start:
            raise ValueError(
                "La date de fin est inférieure à la date de début"
                )
        LOGS.logger.info(
            "Recuperation des CDRS de {customer} du {start} au {stop}".format(
                customer=self.name,
                start=start.strftime('%d%m%Y'),
                stop=stop.strftime('%d%m%Y')
                )
            )
        cdrs = {}
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = (
                "SELECT * FROM cdr "
                "WHERE customer=:name "
                "AND datetime >= :start "
                "AND datetime <= :stop "
                )
            rows = db.query(
                sql,
                name=self.name,
                start='{} {}'.format(start.strftime('%Y-%m-%d'), '00:00:00'),
                stop='{} {}'.format(stop.strftime('%Y-%m-%d'), '23:59:59')
                )
            infos = [
                {
                    'date': row.datetime.date(),
                    'time': row.datetime.strftime('%H%M%S'),
                    'source': row.source,
                    'octets_in': row.octets_in,
                    'octets_out': row.octets_out
                    }
                for row in rows
                ]
            infos = sorted(
                infos,
                key=lambda x: x['date']
                )
            infos = groupby(infos, key=lambda x: x['date'])
            for date, data in infos:
                cdrs[date.strftime('%d%m%Y')] = list(data)
        return cdrs

    def filter_cdrs_policy_wise(self, cdrs):
        """Suivant la politique de bon usage du produit réorganiser les cdrs"""
        result = []
        for policy_name, policy in self.product.policies.items():
            sources = {}
            info = {
                'data_limit': policy['data_limit'],
                'interval': policy['interval'],
                'name': policy_name,
                'total': 0,
                'octets_in': 0,
                'octets_out': 0,
                'details': []
                }
            intervals = []
            for interval in info['interval'].split('_'):
                start, stop, day_start, day_stop = interval.split(',')
                intervals.append(
                    {
                        'start': datetime.strptime(start, '%H%M%S'),
                        'stop': datetime.strptime(stop, '%H%M%S'),
                        'day_start': int(day_start),
                        'day_stop': int(day_stop)
                        }
                    )
            for day, infos in cdrs.items():
                data = []
                for interval in intervals:
                    for i in infos:
                        time = datetime.strptime(i['time'], '%H%M%S')
                        date_iso_weekday = i['date'].isoweekday()
                        start_time = interval['start']
                        stop_time = interval['stop']
                        days_range = range(
                            interval['day_start'],
                            interval['day_stop']+1
                            )
                        in_days = date_iso_weekday in days_range
                        if stop_time >= start_time:
                            in_times = (start_time <= time <= stop_time)
                        else:
                            in_times = (
                                start_time <= time or time <= stop_time
                                )
                        if (in_times and in_days):
                            data.append(i)
                data = sorted(data, key=lambda x: x['source'])
                day_octets_in = sum(
                    [
                        i['octets_in'] for i in data
                        if i['source'] != 'national'
                        ]
                    )
                day_octets_out = sum(
                    [
                        i['octets_out'] for i in data
                        if i['source'] != 'national'
                        ]
                    )
                day_total = day_octets_in + day_octets_out
                data = groupby(data, key=lambda x: x['source'])
                day_details = []
                for source, element in data:
                    if source not in sources:
                        sources[source] = {'octets_in': 0, 'octets_out': 0}
                    element = list(element)
                    source_day_octets_in = sum(
                        [i['octets_in'] for i in element]
                        )
                    source_day_octets_out = sum(
                        [i['octets_out'] for i in element]
                        )
                    source_day_total = (
                        source_day_octets_in + source_day_octets_out
                        )
                    sources[source]['octets_in'] += source_day_octets_in
                    sources[source]['octets_out'] += source_day_octets_out
                    day_details.append(
                        {
                            'source': source,
                            'octets_in': source_day_octets_in,
                            'octets_out': source_day_octets_out,
                            'total': source_day_total
                            }
                        )
                info['total'] += day_total
                info['octets_in'] += day_octets_in
                info['octets_out'] += day_octets_out
                info['details'].append(
                    {
                        'date': day,
                        'octets_in': day_octets_in,
                        'octets_out': day_octets_out,
                        'total': day_total,
                        'details': day_details
                        }
                    )
            info['sources'] = [
                {
                    'source': source,
                    'octets_in': bar['octets_in'],
                    'octets_out': bar['octets_out']
                    }
                for source, bar in sources.items()
                ]
            result.append(info)
        return result

    def apply_product_policies(self, for_real=False):
        """Application des politiques de bon usage du produit pour business"""
        # on vérifie si le client a une configuration particulière
        # pour la date à laquelle on doit commencer à compter
        # sa consommation
        key = 'cdr_start_date:{}'.format(self.name)
        start = REDIS_DB.get(key)  # s'il n'y a pas de clés ce sera None
        if start:
            start = start.decode('utf-8')
            start = datetime.strptime(start + ' 00:00:00', '%d%m%Y %H:%M:%S')
        cdrs = self.get_cdr(start=start)
        cdrs_by_policies = self.filter_cdrs_policy_wise(cdrs)
        for info in cdrs_by_policies:
            total = info['total']
            octets_in = info['octets_in']
            octets_out = info['octets_out']
            info['total'] = round(total / 1000000000, 2)
            info['octets_in'] = round(octets_in / 1000000000, 2)
            info['octets_out'] = round(octets_out / 1000000000, 2)
            url = core.database_url(**FUPDB_CREDENTIALS)
            with records.Database(url) as db:
                sql = (
                    "SELECT status FROM fup "
                    "WHERE customer=:name "
                    "AND policy=:policy "
                    "ORDER BY datetime DESC LIMIT 1"
                    )
                rows = db.query(sql, name=self.name, policy=info['name'])
                row = rows.first()
                if row:
                    status = row.status
                else:
                    status = False
            if info['total'] >= info['data_limit'] and not status:
                info['fup_in'] = True
            elif info['total'] < info['data_limit'] and status:
                info['fup_out'] = True
            if for_real:
                if info.get('fup_in'):  # à limiter
                    url = core.database_url(**FUPDB_CREDENTIALS)
                    with records.Database(url) as db:
                        sql = (
                            "INSERT INTO fup "
                            "(customer, product, policy, datetime, status) "
                            "VALUES "
                            "(:name, :product, :policy, :date_time, :status)"
                            )
                        db.query(
                            sql,
                            name=self.name,
                            product=self.product.name,
                            policy=info['name'],
                            date_time=datetime.now(),
                            status=True
                            )
                    REST_CLIENT.post(
                        '/fup',
                        auth=(
                            CONFIG['AIGUILLIER']['username'],
                            CONFIG['AIGUILLIER']['password']
                            ),
                        params={'customer': self.name}
                        )
                elif info.get('fup_out'):  # libérer
                    url = core.database_url(**FUPDB_CREDENTIALS)
                    with records.Database(url) as db:
                        sql = (
                            "INSERT INTO fup "
                            "(customer, product, policy, datetime, status) "
                            "VALUES "
                            "(:name, :product, :policy, :date_time, :status)"
                            )
                        db.query(
                            sql,
                            name=self.name,
                            product=self.product.name,
                            policy=info['name'],
                            date_time=datetime.now(),
                            status=False
                            )
                    REST_CLIENT.delete(
                        '/fup',
                        auth=(
                            CONFIG['AIGUILLIER']['username'],
                            CONFIG['AIGUILLIER']['password']
                            ),
                        params={'customer': self.name}
                        )
        return cdrs_by_policies

    def get_cdr_summary(self, start=None, stop=None, daily_details=False):
        if not start:
            start = (datetime.now() - timedelta(30))
        if not stop:
            stop = (datetime.now() - timedelta(1))
        cdrs = self.get_cdr(start, stop)
        sorted_cdrs = sorted(
            cdrs,
            key=lambda x: datetime.strptime(x, '%d%m%Y')
            )
        infos = {
            'refnum': self.refnum,
            'name': self.name,
            'consumption': {
                'start': start.strftime('%d%m%Y'),
                'stop': stop.strftime('%d%m%Y'),
                'length': 0,
                'details': []
                }
            }
        for date in sorted_cdrs:
            daily_data = cdrs[date]
            daily_data = sorted(daily_data, key=lambda x: x['source'])
            daily_octets_in = sum([i['octets_in'] for i in daily_data])
            daily_octets_out = sum([i['octets_out'] for i in daily_data])
            daily_total = daily_octets_in + daily_octets_out
            daily_data_by_source = groupby(
                sorted(daily_data, key=lambda x: x['source']),
                key=lambda x: x['source']
                )
            total_by_source = []
            for source, foo in daily_data_by_source:
                foo = list(foo)
                octets_in = sum([i['octets_in'] for i in foo])
                octets_out = sum([i['octets_out'] for i in foo])
                total_by_source.append(
                    {
                        'source': source,
                        'octets_in': octets_in,
                        'octets_out': octets_out,
                        'total': octets_in + octets_out
                        }
                    )
            info = {
                'date': date,
                'total': daily_total,
                'octets_in': daily_octets_in,
                'octets_out': daily_octets_out,
                'details': total_by_source,
                'policy_wise': []
                }
            cdrs_by_policies = self.filter_cdrs_policy_wise({date: daily_data})
            for i in cdrs_by_policies:
                # comme c'est sur une seule journée
                #  - i['details'] est de longueur 1
                #  - i['sources'] et i['details'][0]['details'] sont pareils
                details = i['details'][0]['details']
                del i['details']
                del i['sources']
                if daily_details:
                    i['details'] = details
                info['policy_wise'].append(i)
            infos['consumption']['details'].append(info)
            infos['consumption']['length'] += 1
        LOGS.logger.debug(infos)
        return infos

# EOF
