#!/usr/bin/env python
# encoding: utf-8

import struct
import binascii
# from datetime import datetime
from decimal import Decimal
from .positions import Position
from .devices import Device
from . import REDIS, LOGS
from .gpsdb import GpsDB


class Track(object):

    def __init__(self, device, position):
        self.device = device
        self.position = position

    def json(self):
        return self.__dict__

    def store(self):
        try:
            gpsdb = GpsDB()
            data = {
                "imei": self.device.imei,
                "datetime": self.position.datetime.strftime(
                    '%Y-%m-%d %H:%M:%S %Z'
                    ),
                "longitude": self.position.longitude,
                "latitude": self.position.latitude,
                "angle": self.position.angle,
                "speed": self.position.speed
                }
            gpsdb.insert_track(**data)
            # dbsession = sessionmaker(bind=DB_ENGINE)()
            # dbsession._model_changes = {}
            # track = Track_Table(
            #     imei=self.device.imei,
            #     datetime=self.position.datetime.strftime(
            #         '%Y-%m-%d %H:%M:%S %Z'),
            #     longitude=self.position.longitude,
            #     latitude=self.position.latitude,
            #     angle=self.position.angle,
            #     speed=self.position.speed
            #     )
            # dbsession.add(track)
            # dbsession.commit()
            REDIS.set(self.device.imei, self.position.json())
            return self
        except Exception as exc:
            LOGS.logger.info(exc)
            return exc
        finally:
            LOGS.logger.info("OK")

    def __repr__(self):
        return '<Track {}>'.format(self.__dict__)


class Parser(object):
    """outil d'analyse de datagrammes"""
    def __init__(self, datagram):
        self.datagram = datagram

    def parse(self):
        """à ce stade le parser ne gère qu'un seul type de tracker.
        On devrait pouvoir utiliser un système de plugin dans le futur"""
        self.data_length = int(binascii.hexlify(self.datagram[4:6]), 16)
        # la représentation entière des caractères en 5ème et 6ème position
        # dans le datagramme représente la taille des données contenues
        self.number_of_tracks = int((self.data_length - 10) / 41)
        # pour avoir le nombre de traces, on divise par 41
        # parce qu'on ne prend pas en compte les infos de
        # Driver ID (String(1 à 16)) et Text Message (String(500))
        pattern = '>{fmt}'
        try:
            data = zip(
                *[
                    (
                        struct.unpack(
                            pattern.format(
                                fmt=self.number_of_tracks*'4LlHBIHBHBHs2Hs'
                                ),
                            self.datagram[16:57*self.number_of_tracks]
                        )
                    )[i::17]
                    for i in range(17)
                ]
            )
            # chaque trace compte 17 élements;
            # data contient donc ntracks tuples de 17 éléments.
            result = []
            for track in data:
                position = Position(
                    float(track[0]),
                    Decimal(track[3])/1000000,
                    Decimal(track[4])/1000000,
                    float(track[5]),
                    float(track[10])
                    )
                LOGS.logger.info('Position: {}, {}, {}, {}, {}'.format(
                    position.datetime,
                    position.longitude,
                    position.latitude,
                    position.angle,
                    position.speed
                    )
                )
                device = Device(
                    str(
                        int(binascii.hexlify(self.datagram[8:16]), 16)
                        )
                    )
                LOGS.logger.info(device.imei)
                tracking = Track(device, position)
                LOGS.logger.info("--------Store-------")
                tracking.store()
                result.append(tracking)
            LOGS.logger.info(result)
            return result
        except Exception as e:
            LOGS.logger.info(e)

# EOF
