# -*- coding: utf-8 -*-

from flask import request

from blueflask.lib.infos import success
from blueflask.lib.errors import internal_error, not_found

from ...consts import SERVICE_CODE
from ...core import Client
from . import api

import pika
import json


@api.route('/subscribers/imsi/<string:isdn>', methods=['GET'])
def list_imsi(isdn):
    """
    List IMSI of an ISDN.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: HLR response
        schema:
          id: HLR response
          properties:
            code:
              type: integer
              description: telnet code
            data:
              description: list of infos
              type: array
              items:
                type: string
            date:
              description: date of response
              type: string
            time:
              description: time of response
              type: string
            message:
              description: summary of response
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    client = Client()
    client.list_imsi(isdn)
    code = client.response['code']
    message = client.response['message']
    return_code = '{service_code}-{code}'.format(
        service_code=SERVICE_CODE,
        code=code
        )
    if code == 3001:
        return not_found(
            message=message,
            code=return_code,
            service_code=SERVICE_CODE
            )
    elif code:
        return internal_error(
            message=message,
            code=return_code,
            service_code=SERVICE_CODE
            )
    else:
        return success(service_code=SERVICE_CODE, message=client.response)


@api.route('/subscribers/isdn/<string:imsi>', methods=['GET'])
def list_isdn(imsi):
    """
    List ISDN of an IMSI.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: HLR response
        schema:
          id: HLR response
          properties:
            code:
              type: integer
              description: telnet code
            data:
              description: list of infos
              type: array
              items:
                type: string
            date:
              description: date of response
              type: string
            time:
              description: time of response
              type: string
            message:
              description: summary of response
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    client = Client()
    client.list_isdn(imsi)
    code = client.response['code']
    message = client.response['message']
    return_code = '{service_code}-{code}'.format(
        service_code=SERVICE_CODE,
        code=code
        )
    if code == 3001:
        return not_found(
            message=message,
            code=return_code,
            service_code=SERVICE_CODE
            )
    elif code:
        return internal_error(
            message=message,
            code=code,
            service_code=SERVICE_CODE
            )
    else:
        return success(service_code=SERVICE_CODE, message=client.response)


@api.route('/subscribers/isdn', methods=['PUT'])
def update_isdn():
    """
    Update ISDN.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: HLR response
        schema:
          id: HLR response
          properties:
            code:
              type: integer
              description: telnet code
            data:
              description: list of infos
              type: array
              items:
                type: string
            date:
              description: date of response
              type: string
            time:
              description: time of response
              type: string
            message:
              description: summary of response
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    client = Client()
    data = request.json()
    imsi = data.get('imsi')
    isdn = data.get('isdn')
    if imsi:
        data = {'imsi': imsi}
    elif isdn:
        data = {'isdn': isdn}
    client.update_isdn(**data)
    code = client.response['code']
    message = client.response['message']
    return_code = '{service_code}-{code}'.format(
        service_code=SERVICE_CODE,
        code=code
        )
    if code:
        return internal_error(
            message=message,
            code=return_code,
            service_code=SERVICE_CODE
            )
    return success(service_code=SERVICE_CODE, message=client.response)


@api.route('/subscribers', methods=['POST'])
def create_subscriber():
    """
    Create subscriber in HLR.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      201:
        description: ressource created
        schema:
          id: no content
          properties:
            code:
              type: integer
            message:
              type: string
            info:
              type: string
            status:
              type: string
      409:
        description: conflict
        schema:
          id: conflict
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    auth = request.authorization
    if auth:
        auth = (auth.username, auth.password)
    data = request.get_json()
    params = request.args or {}
    headers = request.headers or {}
    document = {
        'auth': auth or (),
        'params': params,
        'headers': headers,
        'json': data
        }
    try:
        connection = pika.BlockingConnection(
            pika.ConnectionParameters(
                host='localhost',
                virtual_host='/hlr'
                )
            )
    except Exception:
        # exception de connection lancée par pika
        # il faut que rabbitmq-server soit installé
        # et que le vhost /{service} existe déjà
        return internal_error(
            message="serveur RabbitMQ injoignable",
            code='{}-5000'.format(SERVICE_CODE),
            service_code=SERVICE_CODE
            )
    else:
        channel = connection.channel()
        channel.exchange_declare(
            exchange='sendcmd',
            exchange_type='topic'
            )
        # déclaration idempotente: pas grave si l'exchange existe déjà
        channel.basic_publish(
            exchange='sendcmd',
            routing_key='create_subscriber',
            body=json.dumps(document),
            properties=pika.BasicProperties(delivery_mode=2)
            )
        connection.close()
        return success(service_code=SERVICE_CODE)


@api.route('/subscribers', methods=['DELETE'])
def delete_subscriber():
    """
    Delete subscriber in HLR.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: HLR response
        schema:
          id: HLR response
          properties:
            code:
              type: integer
              description: telnet code
            data:
              description: list of infos
              type: array
              items:
                type: string
            date:
              description: date of response
              type: string
            time:
              description: time of response
              type: string
            message:
              description: summary of response
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    client = Client()
    data = request.get_json()
    client.remove_subscriber(**data)
    code = client.response['code']
    message = client.response['message']
    return_code = '{service_code}-{code}'.format(
        service_code=SERVICE_CODE,
        code=code
        )
    if code:
        return internal_error(
            message=message,
            code=return_code,
            service_code=SERVICE_CODE
            )
    return success(service_code=SERVICE_CODE, message=client.response)


@api.route('/subscribers/imsi/<string:service_type>', methods=['PUT'])
def update_subscriber(service_type):
    """
    update subscriber 3G or LTE in HLR.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: HLR response
        schema:
          id: HLR response
          properties:
            code:
              type: integer
              description: telnet code
            data:
              description: list of infos
              type: array
              items:
                type: string
            date:
              description: date of response
              type: string
            time:
              description: time of response
              type: string
            message:
              description: summary of response
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    auth = request.authorization
    if auth:
        auth = (auth.username, auth.password)
    data = request.get_json()
    params = request.args or {}
    headers = request.headers or {}
    document = {
        'auth': auth or (),
        'params': params,
        'headers': headers,
        'json': data,
        'service_type': service_type
        }
    if service_type == 'activate_3g':
        condition = (
          'tplid' not in data or
          'imsi' not in data
          )
    elif service_type == 'activate_lte':
        condition = (
          'tplid' not in data or
          'imsi' not in data or
          'apntplid' not in data
          )
    elif service_type == 'desactivate_lte':
        condition = (
          'imsi' not in data
          )
    else:
        return_code = '{service_code}-{code}'.format(
          service_code=SERVICE_CODE,
          code=400
          )
        return internal_error(
            message='service_type not define',
            code=return_code,
            service_code=SERVICE_CODE
            )
    if condition:
        return_code = '{service_code}-{code}'.format(
          service_code=SERVICE_CODE,
          code=400
          )
        return internal_error(
            message='param incorrect',
            code=return_code,
            service_code=SERVICE_CODE
            )
    else:
        try:
            connection = pika.BlockingConnection(
                pika.ConnectionParameters(
                    host='localhost',
                    virtual_host='/hlr'
                    )
                )
        except Exception:
            # exception de connection lancée par pika
            # il faut que rabbitmq-server soit installé
            # et que le vhost /{service} existe déjà
            return internal_error(
                message="serveur RabbitMQ injoignable",
                code='{}-5000'.format(SERVICE_CODE),
                service_code=SERVICE_CODE
                )
        else:
            channel = connection.channel()
            channel.exchange_declare(
                exchange='sendcmd',
                exchange_type='topic'
                )
            # déclaration idempotente: pas grave si l'exchange existe déjà
            channel.basic_publish(
                exchange='sendcmd',
                routing_key='PUT.hlr',
                body=json.dumps(document),
                properties=pika.BasicProperties(delivery_mode=2)
                )
            connection.close()
            return success(service_code=SERVICE_CODE)

# EOF
