#!/usr/bin/env python
# encoding: utf-8


from flask import g, jsonify, request

from blueflask.lib.errors import bad_request, forbidden, not_found
from blueflask.lib.errors import unauthorized
from blueflask.lib.infos import created, no_content

from ...objects import User, SystemAccount
from ...consts import SERVICE_CODE
from . import api


@api.route('/users', methods=['GET'])
def users():
    """
    Liste des utilisateurs LDAP.
    Renvoie une liste des utilisateurs inscrits dans l'annuaire LDAP
    ---
    tags:
      - Users
    parameters:
      - name: name
        in: query
        type: string
        description: Rechercher par nom
    responses:
      500:
        description: Une erreur interne à l'API s'est produite
      204:
        description: Aucun résultat n'a été trouvé
      200:
        description: Liste des utilisateurs
        schema:
          id: User's location
          properties:
            length:
              type: integer
              description: Nombre d'utilisateurs
            users:
              type: array
              items:
                properties:
                  name:
                    type: string
                    description: Nom de l'utilisateur
                  username:
                    type: string
                    description: Nom d'utilisateur LDAP
    """
    name = request.args.get('name', None)
    if not name:
        users = User.all()
    else:
        users = User.search(name)
    if not users:
        return no_content(service_code=SERVICE_CODE)
    else:
        result = {
            'length': len(users),
            'users': [
                {
                    'name': user.name,
                    'username': user.uid
                    }
                for user in users
                ]
            }
        return jsonify(result)


@api.route('/users', methods=['POST'])
def create_user():
    """
    Créer un nouveau compte utilisateur LDAP.
    ---
    tags:
      - Users
    parameters:
      - name: name
        in: body
        type: string
        description: Nom de l'utilisateur
      - name: email
        in: body
        type: string
        description: Adresse mail de l'utilisateur
      - name: department
        in: body
        type: string
        description: Département de l'utilisateur
      - name: role
        in: body
        type: string
        description: Nom du poste de l'utilisateur
      - name: registration_number
        in: body
        type: string
        description: Numéro matricule de l'utilisateur
    responses:
      500:
        description: Une erreur interne est survenue
      201:
        description: L'utilisateur a été créé
    """
    auth = request.authorization
    if not auth:
        return unauthorized(service_code=SERVICE_CODE)
    else:
        try:
            u = User.verify_auth_token(auth.username)
        except IndexError:
            u = SystemAccount.verify_auth_token(auth.username)
        finally:
            if not u or u.name != 'etech':
                return unauthorized(service_code=SERVICE_CODE)
            elif u.is_blueldap_admin or u.name == 'etech':
                g.user = u.uid
                data = request.get_json()
                condition = (
                    'name' in data and
                    'email' in data and
                    'department' in data and
                    'registration_number' in data and
                    'role' in data
                    )
                if not condition:
                    return bad_request(service_code=SERVICE_CODE)
                else:
                    User.create(
                        data['name'],
                        data['email'],
                        data['department'],
                        data['registration_number'],
                        data['role']
                        )
                    return created(service_code=SERVICE_CODE)
            else:
                return forbidden(service_code=SERVICE_CODE)


@api.route('/users/<string:username>', methods=['PUT'])
def update_user(username):
    """
    Mise à jour des informations d'un utilisateur.
    ---
    tags:
      - Users
    definitions:
      - schema:
          id: Error
          properties:
            error:
              type: string
              description: message court décrivant l'erreur rencontrée
            status:
              type: string
              description: statut de la requête HTTP
            code:
              type: integer
              description: code d'erreur
            message:
              type: string
              description: détails sur l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: username
        in: path
        type: string
        required: true
        description: nom d'utilisateur
    responses:
      500:
        description: Erreur interne à l'API
        $ref: "#/definitions/Error"
      401:
        description: Erreur d'authentification
        $ref: "#/definitions/Error"
      403:
        description: Action non autorisée
        $ref: "#/definitions/Error"
      204:
        description: La mise à jour s'est bien passée
    """
    auth = request.authorization
    if not auth:
        return unauthorized(service_code=SERVICE_CODE)
    else:
        try:
            u = User.verify_auth_token(auth.username)
        except IndexError:
            u = SystemAccount.verify_auth_token(auth.username)
        finally:
            if not u or u.name != 'etech':
                return unauthorized(service_code=SERVICE_CODE)
            elif u.uid != username and not u.is_blueldap_admin:
                return forbidden(service_code=SERVICE_CODE)
            else:
                g.user = u.uid
                try:
                    user = User(username)
                except IndexError:
                    return not_found("{} n'existe pas".format(username))
                else:
                    if not request.get_json():
                        user.reset()
                    else:
                        user.update(**request.get_json())
                    return no_content(service_code=SERVICE_CODE)


@api.route('/users/<string:username>', methods=['DELETE'])
def delete_user(username):
    """
    Suppression d'un utilisateur.
    ---
    tags:
      - Users
    definitions:
      - schema:
          id: Error
          properties:
            error:
              type: string
              description: message court décrivant l'erreur rencontrée
            status:
              type: string
              description: statut de la requête HTTP
            code:
              type: integer
              description: code d'erreur
            message:
              type: string
              description: détails sur l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: username
        in: path
        type: string
        required: true
        description: nom d'utilisateur
    responses:
      500:
        description: Erreur interne à l'API
        $ref: "#/definitions/Error"
      401:
        description: Erreur d'authentification
        $ref: "#/definitions/Error"
      403:
        description: Action non autorisée
        $ref: "#/definitions/Error"
      204:
        description: La suppression s'est bien déroulée
    """
    auth = request.authorization
    if not auth:
        return unauthorized(service_code=SERVICE_CODE)
    else:
        try:
            u = User.verify_auth_token(auth.username)
        except IndexError:
            u = SystemAccount.verify_auth_token(auth.username)
        finally:
            if not u or u.name != 'etech':
                return unauthorized(service_code=SERVICE_CODE)
            elif not u.is_blueldap_admin:
                return forbidden(service_code=SERVICE_CODE)
            else:
                g.user = u.uid
                try:
                    User(username).delete()
                    return no_content(service_code=SERVICE_CODE)
                except IndexError:
                    return not_found(
                        "{} n'existe pas".format(username),
                        service_code=SERVICE_CODE
                        )


@api.route('/users/<string:username>', methods=['GET'])
def user(username):
    """
    Informations sur un utilisateur.
    Retourne toutes les informations d'un utilisateur inscrit dans l'annuaire
    ---
    tags:
      - Users
    definitions:
      - schema:
          id: Error
          properties:
            error:
              type: string
              description: message court décrivant l'erreur rencontrée
            status:
              type: string
              description: statut de la requête HTTP
            code:
              type: integer
              description: code d'erreur
            message:
              type: string
              description: détails sur l'erreur
      - schema:
          id: User
          properties:
            name:
              type: string
              description: Nom de l'utilisateur
            username:
              type: string
              description: Nom d'utilisateur LDAP
            dn:
              type: string
              description: Adresse de l'utilisateur dans l'annuaire
            givenName:
              type: string
              description: Prénom(s) de l'utilisateur
            sn:
              type: string
              description: Nom de l'utilisateur
            mail:
              type: string
              description: Adresse mail de l'utilisateur
            employeeNumber:
              type: string
              description: Matricule de l'utilisateur
            employeeType:
              type: string
              description: Poste de l'utilisateur
            department:
              type: string
              description: Département de l'utilisateur
            mobile:
              type: string
              description: Numéro de téléphone
            telephoneNumber:
              type: string
              description: Numéro de téléphone dans l'entreprise
            manager:
              properties:
                name:
                  type: string
                  description: Nom du manager
                username:
                  type: string
                  description: Nom d'utilisateur LDAP du manager
    parameters:
      - name: username
        in: path
        type: string
        required: true
        description: nom d'utilisateur
    responses:
      500:
        description: Erreur interne à l'API
        $ref: "#/definitions/Error"
      404:
        description: Aucun utilisateur n'a été trouvé
        $ref: "#/definitions/Error"
      401:
        description: Erreur d'authentification
        $ref: "#/definitions/Error"
      403:
        description: Action non autorisée
        $ref: "#/definitions/Error"
      204:
        description: Aucun résultat à retourner
        $ref: "#/definitions/Error"
      200:
        description: Informations sur l'utilisateur
        $ref: "#/definitions/User"
    """
    try:
        user = User(username)
    except IndexError:
        msg = "{} n'existe pas".format(username)
        return not_found(msg, service_code=SERVICE_CODE)
    else:
        infos = {
            key: value
            for key, value in user.__dict__.items()
            if key not in ['userPassword', 'cn', 'handler']
            }
        if hasattr(user, 'manager'):
            infos['manager'] = {
                'name': user.manager.name,
                'uid': user.manager.uid
                }
        try:
            infos['department'] = user.department.name
        except IndexError:
            pass
        return jsonify(infos)

# EOF
