#!/usr/bin/env python
# encoding: utf-8


from flask import jsonify, request

from blueflask.lib.errors import not_found
from blueflask.lib.infos import no_content

from ...objects import Position
from ...consts import SERVICE_CODE
from . import api


@api.route('/positions', methods=['GET'])
def positions():
    """
    Liste des postes disponibles dans l'entreprise.
    Renvoie une liste des postes existants dans l'entreprise
    ---
    tags:
      - Positions
    parameters:
      - name: name
        in: query
        type: string
        description: Rechercher par nom
    responses:
      500:
        description: Une erreur interne à l'API s'est produite
      204:
        description: Aucun résultat n'a été trouvé
      200:
        description: Liste des postes
        schema:
          id: Positions list
          properties:
            length:
              type: integer
              description: Nombre de postes
            positions:
              type: array
              items:
                properties:
                  name:
                    type: string
                    description: Nom du poste
                  department:
                    type: string
                    description: Département du poste
                  dn:
                    type: string
                    description: Adresse du poste dans l'annuaire
    """
    name = request.args.get('name', None)
    if not name:
        positions = Position.all()
    else:
        positions = Position.search(name)
    if not positions:
        return no_content(service_code=SERVICE_CODE)
    else:
        positions = sorted(positions, key=lambda x: x.department.name)
        result = {
            'length': len(positions),
            'positions': [
                {
                    'name': position.name,
                    'department': position.department.name,
                    'dn': position.dn
                    }
                for position in positions
                ]
            }
        return jsonify(result)


@api.route('/positions/<string:name>', methods=['GET'])
def position(name):
    """
    Informations sur un poste.
    Retourne toutes les informations d'un poste
    ---
    tags:
      - Positions
    definitions:
      - schema:
          id: Error
          properties:
            error:
              type: string
              description: message court décrivant l'erreur rencontrée
            status:
              type: string
              description: statut de la requête HTTP
            code:
              type: integer
              description: code d'erreur
            message:
              type: string
              description: détails sur l'erreur
      - schema:
          id: Position
          properties:
            name:
              type: string
              description: Nom du poste
            dn:
              type: string
              description: Adresse du poste dans l'annuaire
            occupants:
              type: array
              items:
                properties:
                  name:
                    type: string
                    description: Nom de l'utilisateur
                  username:
                    type: string
                    description: Nom d'utilisateur LDAP
    parameters:
      - name: name
        in: path
        type: string
        required: true
        description: nom du département
    responses:
      500:
        description: Erreur interne à l'API
        $ref: "#/definitions/Error"
      404:
        description: Aucun utilisateur n'a été trouvé
        $ref: "#/definitions/Error"
      401:
        description: Erreur d'authentification
        $ref: "#/definitions/Error"
      403:
        description: Action non autorisée
        $ref: "#/definitions/Error"
      204:
        description: Aucun résultat à retourner
      200:
        description: Informations sur le poste
        $ref: "#/definitions/Position"
    """
    try:
        position = Position(name)
    except IndexError:
        msg = "{} n'existe pas".format(name)
        return not_found(msg, service_code=SERVICE_CODE)
    else:
        infos = {
            key: value
            for key, value in position.__dict__.items()
            if key not in ['handler', 'ou', 'roleOccupant', 'cn']
            }
        infos['occupants'] = [
            {
                'name': occupant.name,
                'username': occupant.uid
                }
            for occupant in position.occupants
            ]
        return jsonify(infos)

# EOF
