#!/usr/bin/env python
# encoding: utf-8


from flask import g, jsonify, request

from blueflask.lib.errors import bad_request, unauthorized
from blueflask.lib.infos import success, no_content

from ...objects import SystemAccount, User
from ...consts import SERVICE_CODE
from . import api


@api.route('/auth/verify-token', methods=['POST'])
def verify_token():
    """
    Vérification de la validité du jeton de session.
    ---
    tags:
      - Authentication
    definitions:
      - schema:
          id: Error
          properties:
            error:
              type: string
              description: message court décrivant l'erreur rencontrée
            status:
              type: string
              description: statut de la requête HTTP
            code:
              type: integer
              description: code d'erreur
            message:
              type: string
              description: détails sur l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      401:
        description: Authentification incorrecte
        $ref: "#/definitions/Error"
      204:
        description: Authentification réussie
    """
    auth = request.authorization
    if not auth:
        return bad_request(service_code=SERVICE_CODE)
    else:
        try:
            obj = User.verify_auth_token(auth.username)
        except IndexError:
            obj = SystemAccount.verify_auth_token(auth.username)
        if not obj:
            return unauthorized(service_code=SERVICE_CODE)
        else:
            g.user = obj.uid
            return success(obj.uid, service_code=SERVICE_CODE)


@api.route('/auth/request-token', methods=['POST'])
def request_token():
    """
    Demande d'un jeton de session valide 1h.
    ---
    tags:
      - Authentication
    definitions:
      - schema:
          id: Token
          properties:
            token:
              type: string
              descripton: The token
      - schema:
          id: Error
          properties:
            error:
              type: string
              description: message court décrivant l'erreur rencontrée
            status:
              type: string
              description: statut de la requête HTTP
            code:
              type: integer
              description: code d'erreur
            message:
              type: string
              description: détails sur l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      401:
        description: Erreur d'authentification
        $ref: "#/definitions/Error"
      200:
        description: valid token for 1h
        $ref: "#/definitions/Token"
    """
    auth = request.authorization
    try:
        obj = User(auth.username)
    except IndexError:
        try:
            obj = SystemAccount(auth.username)
        except IndexError:
            return unauthorized(service_code=SERVICE_CODE)

    if not obj.authenticate(auth.password):
        return unauthorized(service_code=SERVICE_CODE)
    g.user = obj.uid
    response = {
        'token': obj.generate_auth_token()
        }
    return jsonify(response)


@api.route('/auth', methods=['POST'])
def login():
    """
    Vérifie la validité du couple (userame, password).
    Utile pour ceux/celles qui ne veulent pas générer un jeton de session
    mais veulent juste authentifier en BasicHTTPAuth les utilisateurs.
    ---
    tags:
      - Authentication
    definitions:
      - schema:
          id: Error
          properties:
            error:
              type: string
              description: message court décrivant l'erreur rencontrée
            status:
              type: integer
              description: code d'erreur
            message:
              type: string
              description: détails sur l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      401:
        description: Erreur d'authentification
        $ref: "#/definitions/Error"
      204:
        description: Authentification OK
    """
    auth = request.authorization
    try:
        obj = User(auth.username)
    except IndexError:
        try:
            obj = SystemAccount(auth.username)
        except IndexError:
            return unauthorized(service_code=SERVICE_CODE)

    if not auth or not obj.authenticate(auth.password):
        return unauthorized(service_code=SERVICE_CODE)
    g.user = obj.uid
    return no_content(service_code=SERVICE_CODE)

# EOF
