#!/usr/bin/env python
# encoding: utf-8

import random
import string
import base64
import smtplib
from email.mime.text import MIMEText

import ldap
import requests
from itsdangerous import TimedJSONWebSignatureSerializer as Serializer
from itsdangerous import JSONWebSignatureSerializer as JWS
from itsdangerous import BadSignature, SignatureExpired

from . import consts, core
from .consts import LOGS


__all__ = ["LdapHandler", ]


class LdapHandler:

    def __init__(self):
        pass

    def search(self, object_class, identifier=None, value=None):
        """
        Searches an object inside ldap. When idenfitifier and value are None,
        returns all objects of the given 'object_class'
        Parameters:
            objet_class (str) - "Department", "User", etc.
            identifier (str) - "ou", "sn", "employeeNumber", etc.
            value (str) - value of the identifier
        Return:
            [(object_tree, object_information)]
                object_tree (str) - object tree in ldap
                object_information (dict) - key is utf-8, value is [byte]
        """
        if identifier and value:
            query = (
                "(&" "(objectClass={object_class})" "({identifier}={value})" ")"
            ).format(
                object_class=consts.CONFIG[object_class]["object_class"],
                identifier=identifier,
                value=value
            )
        else:
            query = "(objectClass={object_class})".format(
                object_class=consts.CONFIG[object_class]["object_class"]
            )
        result = core.ldap_conn(
            "search_s",
            consts.CONFIG[object_class]["dn"],
            ldap.SCOPE_SUBTREE,
            filterstr=query,
            attrlist=consts.CONFIG[object_class]["attributes"].split(", "),
        )
        consts.LOGS.logger.info(
            "[LdapHandler]\n"
            "> Search query for '{object_class}' object:\n"
            "-------------------------------------------\n"
            "command: search_s\n"
            "self.dn: {}\n"
            "ldap.SCOPE_SUBTREE: {}\n"
            "filterstr: {}\n"
            "attrlist: {}\n\n"
            "Found {length} occurences\n"
            "{result}\n"
            "-------------------------------------------\n"
            .format(
                consts.CONFIG[object_class]["dn"],
                ldap.SCOPE_SUBTREE,
                query,
                consts.CONFIG[object_class]["attributes"].split(", "),
                object_class=object_class,
                length=len(result),
                result=result,
            )
        )
        return result

    def update(self, object_class, object_tree, param_dict, overwrite=True):
        """
        Update an ldap object. No returned value.
        Parameters:
            objet_class (str) - "Department", "User", etc.
            object_tree (str) - tree of the object to be updated
                eg: cn=user_full_name,ou=people,dc=company
            param_dict (dict) - dict of modifications to insert
            overwrite (bool) - Overwrites related information if True, adds
                a new entry if False
        """
        if overwrite:
            command = ldap.MOD_REPLACE
        else:
            command = ldap.MOD_ADD
        operations = [
            (command, key, value.encode("utf-8"))
            for key, value in param_dict.items()
        ]
        core.ldap_conn("modify_s", object_tree, operations)
        consts.LOGS.logger.info(
            "{object_class} updated.\n"
            "New values inserted: {param_dict}"
            .format(object_class=object_class, param_dict=param_dict)
        )

    def parse_search(self, search_result):
        dn, user_info = search_result

        # [!] this will raise UnicodeDecodeError if removed
        if user_info.get('jpegPhoto'):
            del user_info['jpegPhoto']

        user_info_u = {
            key: value[0].decode("utf-8")
            for key, value in user_info.items()
        }
        return dn, user_info_u

    def generate_password(self):
        """Generates a random password"""
        choices = string.ascii_letters + string.digits
        new_password = "".join(random.sample(choices, random.randint(8, 16)))
        return new_password

    def delete(self, object_class, object_tree, **kwargs):
        """
        Deletes an ldap object. No returned value.
        Parameters:
            objet_class (str) - "Department", "User", etc.
            object_tree (str) - tree of the object to be updated
                eg: cn=user_full_name,ou=people,dc=company
        """
        if object_class == 'User':
            job_tree = kwargs['job_tree']
            command = [(ldap.MOD_DELETE, "roleOccupant", object_tree.encode("utf-8"))]
            core.ldap_conn("modify_s", job_tree, command)
            core.ldap_conn("delete_s", object_tree)
        else:
            raise NotImplementedError
        consts.LOGS.logger.info(
            "{object_class} {object_tree} deleted.\n"
            .format(object_class=object_class, object_tree=object_tree)
        )

    def create(self, object_class, object_tree, **kwargs):
        """
        Creates an ldap object. No returned value.
        Parameters:
            objet_class (str) - "Department", "User", etc.
            object_tree (str) - tree of the object to be updated
                eg: cn=user_full_name,ou=people,dc=company
        """
        if object_class == 'User':
            object_class_b = consts.CONFIG.get(object_class, "object_class").encode('utf-8')
            user_info = kwargs['user_info']
            user_info.append(
                ('objectClass', object_class_b),
            )
            core.ldap_conn("add_s", object_tree, user_info)
        else:
            raise NotImplementedError
        consts.LOGS.logger.info(
            "{object_class} {object_tree} created. \n"
            .format(object_class=object_class, object_tree=object_tree)
        )

    def parse_names(self, name):
        """
        return:
            first_name (str)
            last_name (str)
            full_name (str)
        """
        name_list = name.split()
        if len(name_list) <= 1:
            raise ValueError("'name' must be comprised of a first_name and a last_name.")
        else:
            first_name = name_list[0]
            last_name = name_list[-1]
            full_name = first_name + " " + last_name
        return first_name, last_name, full_name

    def add_to(self, object_class, object_tree, destination_tree):
        """
        Adds an user to a position object. No returned value.
        Parameters:
            objet_class (str) - "Department", "User", etc.
            object_tree (str) - tree of the object to be updated
                eg: cn=user_full_name,ou=people,dc=company
            destination_tree (str) - targeted tree
        """
        if object_class == "User":
            command = [(ldap.MOD_ADD, "roleOccupant", object_tree.encode('utf-8'))]
            try:
                core.ldap_conn("modify_s", destination_tree, command)
                consts.LOGS.logger.info(
                    "User {} added to {} list\n"
                    .format(object_tree, destination_tree)
                )
            except ldap.TYPE_OR_VALUE_EXISTS:
                consts.LOGS.logger.info(
                    "> User already in the position tree\n"
                )
        else:
            raise NotImplementedError

    def remove_from(self, object_class, object_tree, destination_tree):
        """
        Removes an user from a job title object. No returned value.
        Parameters:
            objet_class (str) - "Department", "User", etc.
            object_tree (str) - tree of the object to be updated
                eg: cn=user_full_name,ou=people,dc=company
            destination_tree (str) - targeted tree
        """
        if object_class == "User":
            command = [(ldap.MOD_DELETE, "roleOccupant", object_tree.encode("utf-8"))]
            core.ldap_conn("modify_s", destination_tree, command)
            consts.LOGS.logger.info(
                "User {} removed from {} list\n"
                .format(object_tree, destination_tree)
            )
        else:
            raise NotImplementedError

    def initialize_access(self, email, user_tree):
        """
        Initializes a user password, uidNumber and home directory. Used upon
        user creation.
        Parameters:
            email (str) - user email
            user_tree (str) - tree of the user to be updated
                eg: cn=user_full_name,ou=people,dc=company
        """
        access = {
            "userPassword": self.generate_password(),
            "gidNumber": "501",
            "uidNumber": self._get_next_userid(),
            "homeDirectory": "/home/{}".format(email.split("@")[0]),
            "objectClass": "posixAccount",
        }
        self.update("User", user_tree, access, overwrite=False)
        consts.LOGS.logger.info(
            "User '{}' accesses initialized\n"
            .format(email)
        )
        return access["userPassword"]

    def _get_next_userid(self):
        all_users = self.search("User")
        uid_l = [
            int(employee[1]['uidNumber'][0].decode('utf-8'))
            for employee in all_users
            if employee[1].get('uidNumber')
        ]
        next_uid = max(uid_l) + 1
        return str(next_uid)


class EmailHandler(object):
    """
    Email notifications sent to the user upon account creation or
    password reset.
    """

    def __init__(self):
        self.From = "support_n1@si.blueline.mg"
        self.smtp = smtplib.SMTP("localhost")

    def send(self, to):
        self.text += (
            "Vous pouvez changer votre mot de passe "
            "à l'adresse suivante: {address}\n\n"
            "Pour plus d'informations, merci de contacter "
            "support_n1@si.blueline.mg\n\n"
            "Cordialement\n--\nLDAP Blueline"
        ).format(address=consts.CONFIG.get("LDAP", "RESET_LINK"))
        msg = MIMEText(self.text)
        msg["Subject"] = "Votre compte LDAP Blueline"
        msg["From"] = self.From
        msg["To"] = to
        self.smtp.send_message(msg)

    def create(self, user, password):
        """email sent when account is created"""
        self.text = (
            "Bonjour {name}.\n"
            "Votre compte LDAP blueline a été créé.\n\n"
            "Login: {username}\n"
            "Mot de passe: {password}\n\n"
            "Ce compte LDAP représente votre identité dans "
            "l'annuaire interne de l'entreprise.\n\n"
        ).format(name=user['cn'], username=user['username'], dn=user['dn'], password=password)
        self.send(user['mail'])

    def reset(self, name, username, password, email):
        """email sent when an account's password is reset"""
        self.text = (
            "Bonjour {name}.\n"
            "Votre mot de passe LDAP blueline a été réinitialisé.\n\n"
            "Login: {username}\n"
            "Mot de passe: {password}\n\n"
        ).format(
            name=name,
            username=username,
            password=password
        )
        self.send(email)
# EOF
