#!/usr/bin/env python
# encoding: utf-8

import requests
import logging
from datetime import datetime
from xml.etree import cElementTree as ElementTree

import xmltodict

from .consts import CONFIG, LOGS, AIRTEL_MONEY, REDIS_DB, IzyTV
from .xmlparser import XmlDictConfig


class AirtelMoney(object):

    def __init__(self, data):
        self.response = {
            "code": "046-06-000",
            "status": "000",
            "message": "",
            "response": {},
            "redis": "",
        }
        self.headers = {"content-type" : "application/soap+xml"}
        self.redis_key = "request:payment:{}".format(
            data.get('operation_id') or data.get('order_id')
        )

# payment requests function

    def paymentRequest(self, data):
        """ main function for payment requests """
        try:
            jresponse = self.sendPaymentRequest(data)
            if jresponse:
                self.writeToRedis(data)
            else:
                self.response['redis'] = "Aborted"
        except Exception as e:
            self.response.update({
                "code": "046-06-502",
                "message": str(e),
                "status": 502,
            })
            LOGS.logger.error(str(e))
        return self.response

    def sendPaymentRequest(self, data):
        """
        Sends a transaction payment request to Airtel.
        I tried using the library zeep but somehow airtelmoney's soap schema is
        fucked up and doesn't recognize payment operations. So I am passing everything
        as plain text through requests.
        NB: Transactionidentifier < 50 char or transaction won't pass.
        """
        ref = data['operation_id']
        LOGS.logger.info("[AirtelMoney][{}] Sending request...\n\n".format(ref))
        body = """
          <soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/" xmlns:air="http://www.airtel.com/">
            <soapenv:Header/>
            <soapenv:Body>
              <air:payment>
                <air:APIUsername>{username}</air:APIUsername>
                <air:APIPassword>{password}</air:APIPassword>
                <air:Transactionidentifier>{ref}</air:Transactionidentifier>
                <air:Amount>{amount}</air:Amount>
                <air:Referencenumber>{label}</air:Referencenumber>
                <air:MSISDN>{msisdn}</air:MSISDN>
                <air:MerchantName>{merchant}</air:MerchantName>
              </air:payment>
            </soapenv:Body>
          </soapenv:Envelope>""". format(
            username=AIRTEL_MONEY['payment_username'],
            password=AIRTEL_MONEY['payment_password'],
            merchant=AIRTEL_MONEY['merchant_name'],
            ref=data['operation_id'],
            amount=data["amount"],
            label=data['label'],
            msisdn=data["msisdn"][-9:],
        )
        response = requests.post(
            AIRTEL_MONEY['url_payment'],
            data=body,
            headers=self.headers,
            verify=AIRTEL_MONEY['certificate_payment'],
        )
        LOGS.logger.info(response.text)
        response_text = response.text.replace("air:", "").replace("soapenv:", "")
        root = ElementTree.XML(response_text)
        response_content = XmlDictConfig(root)['Body']['paymentResponse']['paymentResult']
        response_message = response_content.get('message')
        response_status = response_content.get('status')
        LOGS.logger.info(
            "[AirtelMoney][{}]\n"
            "Request status : {}\n"
            "Response: {}\n\n".format(
                ref,
                response_status,
                response_content,
            )
        )
        self.response['status'] = response_status
        self.response['message'] = response_message
        self.response['code'] = "046-06-{}".format(response_status)
        if response_status == "200":
            self.response['response'] = response_content
            return response_content
        else:
            LOGS.logger.error("[AirtelMoney][{}] Request unsuccessful\n\n".format(ref))
            return None

    def writeToRedis(self, data):
        """Writes the initial transaction inside Redis for asynchronous status checks. """
        operation_id = data['operation_id']
        LOGS.logger.info(
            "[AirtelMoney][{}] Writing to Redis...\n\n".format(operation_id)
        )
        today = datetime.today().strftime("%Y-%m-%d %H:%M:%S")
        re_data = {
            "TransactionDate": today,
            "amount": data["amount"],
            "order_id": data["operation_id"],
            "label": data["label"],
            "device_id": data["device_id"],
            "other": data["other"],
            "client_refnum": data["client_refnum"],
            "offre_refnum": data["offre_refnum"],
            "partner_ref": operation_id,
            "operator": "airtel-izytv",
            "caller_num": AIRTEL_MONEY['merchant_name'],
            "service_type": "izytv",
            "status": "En attente de paiement",
            "info": "Activation en attente",
        }
        LOGS.logger.info(re_data)
        re_status = REDIS_DB.hmset(self.redis_key, re_data)
        self.response['redis'] = "Success" if re_status else "Fail"
        LOGS.logger.warning(
            "[AirtelMoney][{}] Status: {}\n\n".format(operation_id, re_status)
        )

# get transaction status functions

    def checkTransactionStatus(self, operation_id):
        """
        Main function for checking the status of a transaction at Airtel.
        Transaction status list is built as :
        {
            <am_status> : [<izytv_txt>, <izytv_code>, <4D_code>,]
        }
        """
        LOGS.logger.info("[AirtelMoney][{}] Sending request...\n\n".format(operation_id))
        body = {
            "COMMAND": {
                "TYPE": "TXNEQREQ",
                "INTERFACEID": "BLUELINEIZY",
                "EXTTRID": operation_id,
            }
        }
        try:
            response = requests.post(
                AIRTEL_MONEY['url_status'],
                data=xmltodict.unparse(body),
                headers=self.headers,
                verify=AIRTEL_MONEY['certificate_status'],
            )
            root = ElementTree.XML(response.text)
            response_content = XmlDictConfig(root)
            response_status = response_content.get('TXNSTATUS')
            LOGS.logger.info(
                "[AirtelMoney][{}]\n"
                "Request status : {}\n"
                "Response: {}\n\n".format(
                    operation_id, response_status, response_content
                )
            )
            self.updateRefPartner(response_content)
            am_status = {
                "TS": ["Transaction validée", 4, "200"],
                "TF": ["Transaction annulée", 3, "000"],
                "TA": ["Transaction annulée", 3, "000"],
                "TIP": ["En attente de paiement", 1, "000"],
            }
            self.response.update({
                "response": response_content,
                "message": am_status.get(response_status)[0] or "Erreur",
                "status": am_status.get(response_status)[1],
                "code": "046-06-{}".format(am_status.get(response_status)[2]),
            })
        except Exception as e:
            LOGS.logger.error(
                "[AirtelMoney][{}] {}\n\n".format(
                    operation_id,
                    str(e)
                )
            )
            self.response.update({
                "code": "046-06-502",
                "message": "Une erreur est survenue: {}".format(str(e)),
                "status": 502,
            })
        LOGS.logger.info(
            "[AirtelMoney]{}\n\n".format(self.response)
        )
        return self.response

    def updateRefPartner(self, status_response):
        """
        For some unknown reasons, we can update directly a key-value dictionnary pair
        from the shell in Redis, but when it goes through our service, it doesn't
        work. So we need to pass again ALL the dictionnary inside Redis to
        overwrite the entry...
        """
        old_entry = REDIS_DB.hgetall(self.redis_key)
        new_entry = {
            "TransactionDate": old_entry[b"TransactionDate"],
            "amount": old_entry[b"amount"],
            "order_id": old_entry[b"order_id"],
            "label": old_entry[b"label"],
            "device_id": old_entry[b"device_id"],
            "other": old_entry[b"other"],
            "client_refnum": old_entry[b"client_refnum"],
            "offre_refnum": old_entry[b"offre_refnum"],
            "caller_num": old_entry[b"caller_num"],
            "partner_ref": status_response.get('TXNID'),
            "status": status_response.get('TXNSTATUS'),
            "operator": "airtel-izytv",
            "service_type": "izytv",
            "info": "Activation en attente",
        }
        REDIS_DB.delete(self.redis_key)
        REDIS_DB.hmset(
            self.redis_key,
            new_entry,
        )
        REDIS_DB.hgetall(self.redis_key)

    def checkTransactionStatusRedis(self, operation_id):
        """Used by 4D in order to check a given transaction current status"""
        val = REDIS_DB.hgetall(self.redis_key.encode('utf-8'))
        data = {key.decode('utf-8'): value.decode('utf-8') for (key, value) in val.items()}
        response = {
            "code": 200,
            "response": {
                "order_id": data["partner_ref"]
            },
            "message": data["info"],
            "status": data["status"]
        }
        return response

# Callback functions

    def callBack(self, transaction):
        """
        Main function called by Airtel to let us know the status of a given transaction.
        Upon reception of the correct parameters, the API will return success to Airtel and
        automatically update transation status inside Redis.

        This service uses authentification and Airtel Callback can't process authentifications
        so we decided to skip this callback process all together and only rely on our cron
        for status checks. These functions are left as-is for future references.
        """
        params = {
            "order_id": transaction['EXTTRID'],
            "status": "Transaction validée" if transaction['TXNSTATUS'] == 200 else "Transaction annulée",
        }
        params["info"] = params["status"]
        LOGS.logger.info("[AirtelMoney]{}\n\n".format(params))
        self.updateStatusRedis(params)
        return {
            "TYPE": "CALLBCKRESP",
            "TXNID": transaction['TXNID'],
            "TXNSTATUS": 200,
            "MESSAGE": "success",
        }

    def updateStatusRedis(self, transaction):
        """Updates a given transaction's status inside Redis."""
        order_id = transaction['order_id']
        LOGS.logger.info(
            "[AirtelMoney]{}\n\n".format(order_id)
        )
        REDIS_DB.hmset(self.redis_key, transaction)

# Transaction reversal functions

    def reverseTransaction(self, transaction):
        """Cancels a transaction at Airtel. Used for cases where an error occured
        on our behalf and the customer must be paid back."""
        body = {
            "COMMAND": {
                "TYPE": "RCGROLLBCK",
                "INTERFACEID": "BLUELINEIZY",
                "TXNID": transaction['operation_id'],
                "USERID": AIRTEL_MONEY['userid'],
            }
        }
        response = requests.post(
            AIRTEL_MONEY['url_reversal'],
            data=xmltodict.unparse(body),
            headers=self.headers,
            verify=AIRTEL_MONEY['certificate_reversal'],
        )
        root = ElementTree.XML(response.text)
        response_content = XmlDictConfig(root)
        response_status = response_content.get('TXNSTATUS')
        self.response.update({
            "status": response_status,
            "message": response_content.get('message'),
            "code": "046-06-{:03d}".format(response_status),
        })
        LOGS.logger.warning(
            "[AirtelMoney][{}]\n"
            "Request status : {}\n"
            "Response: {}\n\n".format(
                transaction['operation_id'],
                response_status,
                response_content
            )
        )
        return self.response
# EOF
