#!/usr/bin/env python
# encoding: utf-8

from systemd import journal

from .consts import HOSTNAME
from .client import Client

import datetime
import json
import threading
import time

import pika
import redis

SERVICE_CODE = '040-04'


def process(smpp_client, queue):
    """Traitement des files d'attentes"""

    def callback(ch, method, properties, body):
        try:
            document = json.loads(body.decode('utf-8'))
        except Exception as exc:
            journal.send(exc)
        else:
            http_verb = document['method']
            http_uri = document['uri']
            if http_verb == 'POST' and http_uri == '/sms':
                if 'msisdn' in document['json']:
                    if ',' in document['json']['msisdn']:
                        # si on a plusieurs destinataires pour le mesage
                        # on remet dans la file d'attente des commandes
                        # pour chaque destinataire
                        msisdns = document['json']['msisdn'].replace(' ', '')
                        msisdns = msisdns.split(',')
                        for index, msisdn in enumerate(msisdns):
                            bar = {
                                'message': document['json'].get(
                                    'message',
                                    'message vide'
                                    ),
                                'msisdn': msisdn,
                                'sender_id': document['json'].get(
                                    'sender_id',
                                    'bip'
                                    ),
                                'encoding': document['json'].get(
                                    'encoding',
                                    'DEFAULT'
                                    )
                                }
                            data = {
                                key: value for key, value in document.items()
                                if key != 'json'
                                }
                            data['json'] = bar
                            data['index'] = index
                            ch.basic_publish(
                                exchange='async_requests',
                                routing_key=queue,
                                body=json.dumps(data),
                                properties=pika.BasicProperties(
                                    delivery_mode=2
                                    )
                                )
                    else:
                        # on envoie sur la file d'attente la commande
                        # d'envoi de SMS
                        msisdn = document['json']['msisdn']
                        message = document['json'].get(
                            'message',
                            'message vide'
                            )
                        sender_id = document['json'].get('sender_id', 'bip')
                        request_id = document['headers']['X-Request-Id']
                        if not HOSTNAME.startswith('blpr'):
                            status_code = 200
                            result = {
                                'msisdn': msisdn,
                                'date': datetime.datetime.now().strftime(
                                    '%d-%m-%Y %H:%M:%S'
                                    ),
                                'submit': True
                                }
                        else:
                            log_msg = (
                                "POST smsc: tentative d'envoi de sms vers {}"
                                ).format(msisdn)
                            journal.send(
                                log_msg,
                                REQUEST_ID=request_id
                                )
                            try:
                                smpp_client.send_sms(
                                    msisdn,
                                    message,
                                    request_id=request_id,
                                    sender_id=sender_id
                                    )
                            except Exception as exc:
                                log_msg = (
                                    "Erreur survenue lors de l'envoi: {}"
                                    ).format(exc)
                                journal.send(
                                    log_msg,
                                    REQUEST_ID=request_id
                                    )
                                status_code = 400
                                result = {
                                    'code': SERVICE_CODE+'-4000',
                                    'msisdn': msisdn,
                                    'message': exc,
                                    'date': datetime.now().strftime(
                                        '%d-%m-%Y %H:%M:%S'
                                        )
                                    }
                            else:
                                log_msg = (
                                    "POST smsc: delivre au smsc "
                                    "pour envoi vers {}"
                                    ).format(msisdn)
                                journal.send(
                                    log_msg,
                                    REQUEST_ID=request_id
                                    )
                                if smpp_client.infos['submit']:
                                    result = smpp_client.infos
                                    status_code = 200
                                    result['code'] = SERVICE_CODE+'-2000'
                                    result['message'] = 'ok'
                                else:
                                    if smpp_client.infos['submit'] is None:
                                        code = SERVICE_CODE+'-5004'
                                        status_code = 504
                                    else:
                                        code = SERVICE_CODE+'-5003'
                                        status_code = 503
                                    message = smpp_client.infos['message']
                                    date = smpp_client.infos['date']
                                    result = {
                                        'code': code,
                                        'msisdn': msisdn,
                                        'message': message,
                                        'date': date
                                        }
                        infos = {
                            'status': status_code,
                            'content': result
                            }
                        key = 'smsc:{request_id}'.format(
                            request_id=request_id
                            )
                        if 'index' in document:
                            key += ':{}'.format(document['index'])
                        REDIS_SERVER = redis.Redis()
                        REDIS_SERVER.hmset(key, infos)
                        time.sleep(0.25)
        ch.basic_ack(delivery_tag=method.delivery_tag)
    connection = pika.BlockingConnection(
        pika.ConnectionParameters(
            host='localhost',
            virtual_host='/smsc'
            )
        )
    channel = connection.channel()
    channel.basic_qos(prefetch_count=1)
    channel.basic_consume(callback, queue=queue)
    channel.start_consuming()


class Worker(threading.Thread):

    def __init__(self, name, queue):
        threading.Thread.__init__(self)
        self.name = name
        self.queue = queue
        self.smpp_client = Client()

    def run(self):
        process(self.smpp_client, self.queue)

# EOF
