#!/usr/bin/env python
# encoding: utf-8


from flask import jsonify, request

from blueflask.lib.decorators import auth as ldap
from blueflask.lib.infos import no_content, success
from blueflask.lib.errors import not_found
from ...consts import SERVICE_CODE
from .. import errors
from . import api

import json

import pika
import redis

REDIS_SERVER = redis.Redis()


@api.route('/async/<path:uri>', methods=['GET', 'POST', 'PUT', 'DELETE'])
@ldap.login_required
def async_call_uri(uri):
    """
    Appels asynchrones.
    Endpoint permettant d'appeler de façon asynchrone une uri.
    ---
    tags:
      - async
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: uri
        in: path
        required: true
        description: Chemin relatif vers la ressource à appeler
        schema:
          type: string
    responses:
      200:
        description: la requête a été placée en file d'attente
      500:
        description: le serveur RabbitMQ local n'est pas joignable
    """
    auth = request.authorization
    if auth:
        auth = (auth.username, auth.password)
    document = {
        'method': request.method,
        'service': 'smsc',
        'version': 1,
        'uri': '/{}'.format(uri),
        'auth': auth or (),
        'params': request.args or {},
        'headers': request.headers or {},
        'json': request.get_json() or {}
        }
    try:
        connection = pika.BlockingConnection(
            pika.ConnectionParameters(
                host='localhost',
                virtual_host='/smsc'
                )
            )
    except Exception:
        # exception de connection lancée par pika
        # il faut que rabbitmq-server soit installé
        # et que le vhost /{service} existe déjà
        return errors.error(5000)
    else:
        channel = connection.channel()
        channel.exchange_declare(
            exchange='async_requests',
            exchange_type='topic'
            )
        # déclaration idempotente: pas grave si l'exchange existe déjà
        channel.basic_publish(
            exchange='async_requests',
            routing_key='{method}.smsc'.format(
                method=request.method
                ),
            body=json.dumps(document),
            properties=pika.BasicProperties(delivery_mode=2)
            )
        connection.close()
        return success(service_code=SERVICE_CODE)


@api.route('/requests', methods=['GET'])
@ldap.login_required
def get_all_existing_requests():
    """
    Liste des requêtes.
    Cette ressource permet d'avoir la liste des request-id dont
    le traitement est terminé.
    ---
    tags:
      - async
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: liste des request-id
      204:
        description: aucun request-id trouvé
    """
    keys = REDIS_SERVER.keys('smsc:*')
    if not keys:
        return no_content(service_code=SERVICE_CODE)
    else:
        request_ids = set([i.decode('utf-8').split(':')[1] for i in keys])
        result = {
            'requests': list(request_ids),
            'length': len(request_ids)
            }
        return jsonify(result)


@api.route('/requests/<string:request_id>', methods=['GET'])
@ldap.login_required
def get_request_response(request_id):
    """
    Résultat d'une requête.
    Cette ressource renvoie la réponse correspondant à la requête
    identifée par le request-id passé dans l'URL
    ---
    tags:
      - async
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: réponse à la requête
      404:
        description: ce request-id n'existe pas
    """
    keys = REDIS_SERVER.keys('smsc:{}*'.format(request_id))
    if not keys:
        return not_found(service_code=SERVICE_CODE)
    else:
        result = {'status': []}
        statuses = {}
        for key in keys:
            response = REDIS_SERVER.hgetall(key)
            content = eval(response[b'content'])
            code = content['code']
            if code not in statuses:
                statuses[code] = [content]
            else:
                statuses[code].append(content)
        for code, data in statuses.items():
            infos = {
                'code': code,
                'message': data[0]['message'],
                'msisdn': ','.join([i['msisdn'] for i in data])
                }
            result['status'].append(infos)
        return jsonify(result)

# EOF
