#!/usr/bin/env python
# encoding: utf-8

from datetime import datetime, timedelta

from .consts import CONFIG
from .consts import LOGS

import records

import smpplib.client
import smpplib.command
import smpplib.consts
import smpplib.exceptions
import smpplib.smpp
import smpplib.gsm
import sys
import requests
import re

__all__ = [
    'Getsms',
    ]


class Getsms(object):

    def __init__(self, name='default'):
        self.msisdn = str(name)
        smpp_config = 'GETSMS_{}'.format(str(name))
        self.host = CONFIG[smpp_config]['host']
        self.port = CONFIG[smpp_config]['port']
        self.username = CONFIG[smpp_config]['username']
        self.password = CONFIG[smpp_config]['password']
        self.api_url = CONFIG['server']['host']
        self.esb_url = CONFIG['server']['host_esb']
        self.list_msisdn = CONFIG['GET_SMS']['msisdn'].split(',')
        self.list_msisdn_email = CONFIG['GET_SMS']['msisdn_email'].split(',')
        self.email_from = CONFIG[smpp_config]['email_from']
        self.email_to = CONFIG[smpp_config]['email_to']
        self.title = CONFIG[smpp_config]['title']
        LOGS.logger.info(self.msisdn)

    def all(msisdn):
        name = '{}'.format(str(msisdn))
        return [Getsms(name)]

    def verify_token(self, auth):
        api_url = '{}/auth/verify-token'.format(
            self.api_url
            )
        result = requests.post(
            api_url,
            auth=auth
            )
        return result

    def connect_receiver_sms(self):
        self.smpp_client = smpplib.client.Client(
            self.host,
            int(self.port)
            )
        self.smpp_client.set_message_sent_handler(
            lambda pdu: sys.stdout.write('sent {} {}\n'.format(
                pdu.sequence,
                pdu.message_id
                )
            )
            )
        self.smpp_client.set_message_received_handler(self.callback_response)
        try:
            self.smpp_client.connect()
        except smpplib.exceptions.ConnectionError:
            raise SystemError("Impossible de se connecter au SMSC")
        else:
            try:
                self.smpp_client.bind_transceiver(
                    system_id=self.username,
                    password=self.password
                    )
                from threading import Thread
                t = Thread(target=self.listen)
                t.start()
            except smpplib.exceptions.PDUError:
                result = {
                        'code': '301',
                        'message': (
                            'Impossible de se reconnecter '
                            'en bind_transceiver au SMSC')
                        }
            else:
                result = {
                        'code': '200',
                        'message': ('connecter to bind_transceiver')
                        }
            return result

    def callback_response(self, pdu):
        """Si le SMSC a envoyé le sms on enregistre """
        """le status du sms dans une tabel"""
        msisdn_dest = pdu.destination_addr.decode("utf-8")
        msisdn_init = pdu.source_addr.decode("utf-8")
        message = pdu.short_message.decode("latin1")
        bool_email = False
        cond = (
            msisdn_dest is not None
            or msisdn_dest != 'None'
            )
        if cond and str(msisdn_dest) in self.list_msisdn:
            message = re.findall(
                    r"[A-Za-z0-9 \\r\\nÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖØŒŠþ"
                    "ÙÚÛÜÝŸàáâãäåæçèéêëìíîïðñòóôõöøœšÞùúûüýÿ!\"#$%"
                    "&'()+,\-./:;<=>?¡¿^{}\\\[~\]|\u20AC]+", message
                    )
            message = ''.join(message)
            kwargs = {
                'engine': CONFIG['db']['engine'],
                'user': CONFIG['db']['user'],
                'password': CONFIG['db']['password'],
                'host': CONFIG['db']['host'],
                'port': CONFIG['db']['port'],
                'name': CONFIG['db']['database']
                }
            url = self.database_url(**kwargs)
            with records.Database(url) as db:
                sql = (
                    "INSERT INTO smpp_get_message "
                    "(create_date, msisdn_init, msisdn_dest, "
                    "message, bool_sms) VALUES(:create_date, "
                    ":msisdn_init, :msisdn_dest, :message, :bool_sms)"
                    )
                LOGS.logger.info(sql)
                db.query(
                    sql,
                    create_date=datetime.now(),
                    message=str(message),
                    msisdn_init=msisdn_init,
                    bool_sms='False',
                    msisdn_dest=msisdn_dest
                    )
            if str(msisdn_dest) in self.list_msisdn_email:
                params = {
                    "mail_to": self.email_to,
                    "mail_from": self.email_from,
                    "mail_text": message,
                    "title": self.title
                    }
                email_url = '{}/api/send/email'.format(
                    self.esb_url
                    )
                result = requests.post(email_url, params=params)
                if result.status_code == 200:
                    bool_email = True
            msg = '<<< msg={} status={} msisdn={} email={}'.format(
                message,
                pdu.status,
                msisdn_init,
                bool_email
                )
            LOGS.logger.info(msg)

    def disconnect_receiver_sms(self):
        smpp_client = smpplib.client.Client(
            self.host,
            int(self.port)
            )
        try:
            smpp_client.__del__()
        except smpplib.exceptions.ConnectionError:
            raise SystemError("Impossible de se déconnecter au SMSC")

    def listen(self):
        while True:
            try:
                self.smpp_client.listen()
            except Exception as exc:
                print(self.msisdn, exc)

    def database_url(self, **kwargs):
        url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
            eng=kwargs.get('engine'),
            user=kwargs.get('user'),
            password=kwargs.get('password'),
            host=kwargs.get('host'),
            port=kwargs.get('port'),
            name=kwargs.get('name')
            )
        return url

    def chunks(self, l, n):
        if n < 1:
            n = 1
        return [l[i:i + n] for i in range(0, len(l), n)]

    def gsm_decode(self, hexstr):
        hexparts = self.chunks(hexstr, 2)
        res = ''
        for byte in hexparts:
            byte = int(byte, 16)
            if byte <= len(smpplib.gsm.GSM_CHARACTER_TABLE):
                res += smpplib.gsm.GSM_CHARACTER_TABLE[byte]
            else:
                res += ' '
        return res

    def delivery_sms(self, request_id, msisdn=None):
        LOGS.logger.info(
            "Recuperation le status des sms "
            "du {msisdn} request_id {request_id}".format(
                msisdn=msisdn,
                request_id=request_id
                )
            )
        sms = {'nbr': 0}
        kwargs = {
            'engine': CONFIG['db']['engine'],
            'user': CONFIG['db']['user'],
            'password': CONFIG['db']['password'],
            'host': CONFIG['db']['host'],
            'port': CONFIG['db']['port'],
            'name': CONFIG['db']['database']
            }
        url = self.database_url(**kwargs)
        infos = []
        with records.Database(url) as db:
            if msisdn is not None:
                sql = (
                    "SELECT * FROM get_delivery_sms "
                    "WHERE request_id = :request_id "
                    "AND msisdn = :msisdn "
                    )
                rows = db.query(
                    sql,
                    request_id=str(request_id),
                    msisdn=str(msisdn[-9:])
                    )
            else:
                sql = (
                    "SELECT * FROM get_delivery_sms "
                    "WHERE request_id = :request_id "
                    )
                rows = db.query(
                    sql,
                    request_id=str(request_id)
                    )
            for row in rows:
                if row.code == '000':
                    code = '200'
                else:
                    code = '500'
                infos.append({
                        'date': row.create_date,
                        'submit_date': row.submit_date,
                        'done_date': row.done_date,
                        'status': row.status,
                        'code': code,
                        'msisdn_dest': row.msisdn,
                        'msisdn_source': row.msisdn_source
                        })
            nbr = len(infos)
            infos = sorted(
                infos,
                key=lambda x: x['date']
                )
            for i in infos:
                del i['date']
            sms['sms'] = list(infos)
            sms['nbr'] = nbr
        LOGS.logger.info("Recuperation de {} sms".format(nbr))
        return sms

    def check_sms(
            self,
            start=None,
            stop=None,
            bool_sms=None
            ):
        if not start:
            start = (datetime.now() - timedelta(30))
        if not stop:
            stop = (datetime.now() - timedelta(1))
        if stop < start:
            raise ValueError(
                "La date de fin est inférieure à la date de début"
                )
        LOGS.logger.info(
            "Recuperation des sms "
            "du {start} au {stop} status {bool_sms} du {number}".format(
                start=start.strftime('%Y%m%d %H%M%S'),
                stop=stop.strftime('%Y%m%d %H%M%S'),
                bool_sms=bool_sms,
                number=self.msisdn
                )
            )
        sms = {'nbr': 0}
        kwargs = {
            'engine': CONFIG['db']['engine'],
            'user': CONFIG['db']['user'],
            'password': CONFIG['db']['password'],
            'host': CONFIG['db']['host'],
            'port': CONFIG['db']['port'],
            'name': CONFIG['db']['database']
            }
        url = self.database_url(**kwargs)
        infos = []
        with records.Database(url) as db:
            if not bool_sms:
                sql = (
                    "SELECT * FROM smpp_get_message "
                    "WHERE create_date >= :start "
                    "AND create_date <= :stop "
                    "AND msisdn_dest = :msisdn_dest "
                    )
                rows = db.query(
                    sql,
                    start='{}'.format(
                        start.strftime('%Y%m%d %H%M%S')
                        ),
                    stop='{}'.format(
                        stop.strftime('%Y%m%d %H%M%S')
                        ),
                    msisdn_dest=str(self.msisdn)
                    )
            else:
                sql = (
                    "SELECT * FROM smpp_get_message "
                    "WHERE bool_sms=:bool_sms "
                    "AND to_char(create_date, 'YYYY-MM-DD HH24:MI:SS') "
                    "BETWEEN :start AND :stop "
                    "AND msisdn_dest = :msisdn_dest "
                    )
                rows = db.query(
                    sql,
                    bool_sms=bool_sms,
                    start='{}'.format(
                        start.strftime('%Y%m%d %H%M%S')
                        ),
                    stop='{}'.format(
                        stop.strftime('%Y%m%d %H%M%S')
                        ),
                    msisdn_dest=str(self.msisdn)
                    )
            infos = [
                {
                    'id': row.id,
                    'date': row.create_date.strftime('%Y%m%d'),
                    'time': row.create_date.strftime('%H%M%S'),
                    'msisdn_init': row.msisdn_init.strip(),
                    'msisdn_dest': row.msisdn_dest.strip(),
                    'bool_sms': row.bool_sms,
                    'message': str(row.message)
                    }
                for row in rows
                ]
            nbr = len(infos)
            infos = sorted(
                infos,
                key=lambda x: x['date']
                )
            sms['sms'] = list(infos)
            sms['nbr'] = nbr
        LOGS.logger.info("Recuperation de {} sms".format(len(sms)))
        return sms

    def update_sms(self, ident):
        LOGS.logger.info(
            "Update bool_sms de id {ident}".format(
                ident=ident)
            )
        sms = {
            'id': ident,
            }
        kwargs = {
            'engine': CONFIG['db']['engine'],
            'user': CONFIG['db']['user'],
            'password': CONFIG['db']['password'],
            'host': CONFIG['db']['host'],
            'port': CONFIG['db']['port'],
            'name': CONFIG['db']['database']
            }
        url = self.database_url(**kwargs)
        with records.Database(url) as db:
            sql = (
                "UPDATE smpp_get_message "
                "SET bool_sms =:bool_sms "
                "WHERE id=:ident"
                )
            result = db.query(
                sql,
                bool_sms=True,
                ident=ident,
                )
            LOGS.logger.info(
                "Update bool_sms de id {ident} result:{res}".format(
                    ident=ident,
                    res=str(result.pending)
                    )
                )
            sms['bool_sms'] = str(result.pending)
        return sms


# EOF
