#!/usr/bin/env python
# encoding: utf-8

from systemd import journal

from .consts import APIGW, LOGS, SERVICE_CODE
from .client import Client

import datetime
import json
import threading
import time

import pika
import requests


def process(smpp_client, queue):
    """Traitement des files d'attentes"""

    def callback(ch, method, properties, body):
        try:
            document = json.loads(body.decode('utf-8'))
        except Exception as exc:
            LOGS.logger.error(exc)
            ch.basic_ack(delivery_tag=method.delivery_tag)
        else:
            if 'msisdn' not in document['json']:
                ch.basic_ack(delivery_tag=method.delivery_tag)
            else:
                response = {
                    'code': '{}-200'.format(SERVICE_CODE),
                    'status': 200,
                    'message': 'OK'
                    }
                msisdn = document['json']['msisdn']
                message = document['json'].get(
                    'message',
                    'message vide'
                    )
                sender_id = document['json'].get('sender_id', 'bip')
                request_id = document['headers']['X-Request-Id']
                log_msg = (
                    "POST smsc: tentative d'envoi de sms vers {}"
                    ).format(msisdn)
                journal.send(
                    log_msg,
                    REQUEST_ID=request_id
                    )
                try:
                    # international
                    if document['json'].get("intl"):
                        smpp_client.send_sms(
                            msisdn,
                            message,
                            intl=True,
                            request_id=request_id,
                            sender_id=sender_id
                            )
                    else:
                        smpp_client.send_sms(
                            msisdn,
                            message,
                            request_id=request_id,
                            sender_id=sender_id
                            )
                except Exception as exc:
                    log_msg = (
                        "Erreur survenue lors de l'envoi: {}"
                        ).format(exc)
                    journal.send(
                        log_msg,
                        REQUEST_ID=request_id
                        )
                    ch.basic_ack(delivery_tag=method.delivery_tag)
                    response['status'] = 500
                    response['code'] = SERVICE_CODE+'-5000'
                    response['error'] = 'Erreur interne'
                    response['message'] = {
                        'msisdn': msisdn,
                        'date': datetime.datetime.now().strftime(
                            '%d-%m-%Y %H:%M:%S'
                            ),
                        'error': exc
                        }
                else:
                    log_msg = (
                        "POST smsc: delivre au smsc "
                        "pour envoi vers {}"
                        ).format(msisdn)
                    journal.send(
                        log_msg,
                        REQUEST_ID=request_id
                        )
                    ch.basic_ack(delivery_tag=method.delivery_tag)
                    response['message'] = smpp_client.infos
                    if smpp_client.infos['submit']:
                        response['code'] = SERVICE_CODE+'-200'
                    else:
                        if smpp_client.infos['submit'] is None:
                            response['code'] = SERVICE_CODE+'-5004'
                            response['status'] = 504
                        else:
                            response['code'] = SERVICE_CODE+'-5003'
                            response['status'] = 503
                infos = {i: j for i, j in response.items()}
                infos['service'] = 'smsc'
                infos['request_id'] = request_id
                if 'index' in document:
                    infos['index'] = document['index']
                requests.post(
                    'https://{fqdn}/requests'.format(fqdn=APIGW),
                    json=infos
                    )
                time.sleep(0.75)

    connection = pika.BlockingConnection(
        pika.ConnectionParameters(
            host='localhost',
            virtual_host='/smsc'
            )
        )
    channel = connection.channel()
    channel.basic_qos(prefetch_count=1)
    channel.basic_consume(callback, queue=queue)
    channel.start_consuming()


class Worker(threading.Thread):

    def __init__(self, name, queue):
        threading.Thread.__init__(self)
        self.name = name
        self.queue = queue
        self.smpp_client = Client(self.queue)

    def run(self):
        process(self.smpp_client, self.queue)

# EOF
