# -*- coding: utf-8 -*-

"""
Ce module contient une classe qui permet d'appeler une ressource dans une API,
et une autre qui permet de traduire le payload XML d'une requête SOAP en
requête REST pour une API.

Une API est définie par:
  - un nom de service
  - un nom de sous-service (optionel)
  - un numéro de version

Une requête REST ressemble donc à ceci:
  HTTP_VERB /service{/sous-service}/vversion/ressource
"""

from lxml import etree
from spyne.model.fault import Fault

from rest_client import RestClient
from .consts import ADDRESS
from .consts import API_REGISTRY
from .consts import CONFIG
from .consts import HOSTNAME
from .consts import HTTP_VERBS
from .consts import REQUESTS_LOGS

import datetime
import re
import xml

import requests
import xmltodict


__all__ = [
    'BadRequestError',
    'Api',
    'SoapPayload'
]


def verify_token(source, token, **kwargs):
    """Vérifie avec le serveur LDAP si le token d'authentification
    est toujours valide.

    La fonction vérifie d'abord dans le registre d'APIs
    si le token utilisé n'a pas déjà été vérifié par un service de confiance.
    Si ce n'est pas le cas, alors une vérification est effectuée auprès de
    l'annuaire LDAP et le résultat est stocké dans le registre d'APIs."""

    key = 'token:{}'.format(token)
    env = kwargs.get('env', None)
    service = kwargs.get('service')
    path = kwargs.get('path', '')
    headers = kwargs.get('headers', {'X-Real-Ip': ADDRESS})
    ldap_api = Api('ldap', env=env)
    username = None

    if service == 'ldap' and path.startswith('/auth'):
        return True  # on laisse passer la requête telle quelle
    elif source not in API_REGISTRY:
        # la requête provient d'une source extérieure
        # au cluster de services
        # donc on vérifie le token
        result = API_REGISTRY.get(key)
        if not result:  # on demande une vérification de token
            response = ldap_api.call(
                'POST',
                '/auth/verify-token',
                auth=(token, ''),
                headers=headers
                )
            if response['status'] != 200:  # token invalide
                return False
            else:
                # on récupère les noms d'utilisateurs correspondant à ce token
                user = response['content']['message']
                username = min(user, key=lambda x: len(x))
                if len(username) > 20:
                    username = username[:20]
                API_REGISTRY.set(key, username)
                API_REGISTRY.expire(key, 3600)
                # on force l'enregistrement dans redis à expirer
                # au bout d'une heure
                return username

        else:
            # le token a déjà été vérifié une fois
            username = result.decode('utf-8')
            return username
    else:
        return True


class BadRequestError(Fault):

    """Erreur SOAP"""

    def __init__(self):
        super(BadRequestError, self).__init__(
            faultcode='Client.BadRequestError',
            faultstring='Check your input xml'
        )


class Api(object):

    """Représente une API enregistrée sur ce gateway"""

    def __init__(self, service, version=None, env=None):
        """Le constructeur accepte quatre variables:
              - une désignant le service
              - une désignant l'environnement du service
              - une désignant la version du service (au format vX)
        """
        registry_key = 'service:{service}'.format(service=service)
        self.env = env
        if not self.env:
            infos = API_REGISTRY.hgetall(registry_key)
        elif not HOSTNAME.startswith('api'):
            # ça veut dire qu'on essaye d'appeler une API
            # à partir d'un serveur autre que api.blueline.mg
            # en précisant l'environnement
            raise ValueError(
                "Inutile de préciser {}".format(self.env)
                )
        else:
            # on a précisé un environnement
            # et on se trouve sur api.blueline.mg
            # donc on va lire le registre de l'environnement
            import redis
            registry = redis.Redis(
                host=CONFIG['API_REGISTRIES'][self.env],
                db=15
                )
            infos = registry.hgetall(registry_key)

        if not infos:
            # le service n'est pas déclaré dans le registre
            raise KeyError(
                "Le service '{}' n'est pas déclaré".format(service)
                )
        else:
            self.service = service

            for key, value in infos.items():
                # chaque info contenu dans la base redis pour ce service
                # devient un attribut
                # les attributs qu'un service déclare sont les suivants:
                #  - host
                #  - service_code
                #  - version
                setattr(self, key.decode('utf-8'), value.decode('utf-8'))

            if not version:
                if not hasattr(self, 'version'):
                    self.version = 'v1'
                else:
                    self.version = 'v{}'.format(self.version)
                    # dans le registre la version est déclarée comme un entier
            else:
                self.version = 'v{}'.format(self.version)

            if not hasattr(self, 'host'):
                raise ValueError(
                    "Le service {} n'a aucun hôte déclaré".format(service)
                    )
            else:
                self.host = self.host.split('.')[0]

            if not hasattr(self, 'service_code'):
                raise ValueError(
                    "Le service {} n'a aucun 'service_code'".format(service)
                    )

            self.location = (
                'https://{host}.blueline.mg:{port}'.format(
                    host=self.host,
                    port=int(self.service_code[1:].replace('-', ''))
                    )
                )

    def spec(self):
        """Spécifications OpenAPI"""
        response = requests.get(self.location + '/api/spec')
        return response.json()

    def call(self, method, ressource, **kwargs):
        """Appel d'une ressource de l'API"""
        headers = kwargs.get('headers')
        auth = kwargs.get('auth')
        params = kwargs.get('params')
        json_doc = kwargs.get('json')
        url = '{location}/api/{version}'.format(
            location=self.location,
            version=self.version
            )
        # avant d'appeler une ressource, on doit vérifier le token
        try:
            token = auth[0]
        except (TypeError, IndexError):
            # la requête est incorrecte, il faut quelque chose
            # dans le couple (username, password)
            username = False
        else:
            username = verify_token(
                headers['X-Real-Ip'],
                token,
                service=self.service,
                path=ressource,
                env=self.env
                )
        if not username:
            # le token n'a pas pu être vérifié, il faut renvoyer un HTTP 401
            result = {
                'status': 401,
                'content': {
                    'code': '04-401',
                    'error': 'authentification requise',
                    'message': (
                        'Une authentification est nécessaire '
                        'pour accéder à la ressource'
                        ),
                    'status': 401
                    },
                'headers': headers
                }
            self.record_result(method, url, result)
            return result
        elif username and not isinstance(username, bool):
            auth = (username, auth[1])
        client = RestClient(url)
        result = client.__call_url__(
            method,
            ressource,
            auth=auth,
            headers=headers,
            params=params,
            json=json_doc
        )
        self.record_result(method, url, result)
        content = result['content']
        if isinstance(content, dict) and result['status'] == 500:
            message = content['message']
            if isinstance(message, dict):
                content['message'] = (
                    'une erreur interne est survenue; '
                    'veuillez contacter support_n1@si.blueline.mg'
                    )
        return result

    def record_result(self, method, url, result):
        timestamp = datetime.datetime.now().strftime('%d%m%Y%H%M%S')
        content = result['content']
        if isinstance(content, dict):
            infos = {
                key: value
                for key, value in content.items()
                }
        else:
            infos = {'message': content}
        infos['request-method'] = method
        infos['request-url'] = url
        key = 'request:{service}:{request_id}:{timestamp}'.format(
            service=self.service,
            request_id=result['headers']['X-Request-Id'],
            timestamp=timestamp
            )
        REQUESTS_LOGS.hmset(key, infos)
        REQUESTS_LOGS.expire(key, 2678400)


class SoapPayload(object):

    """Représente un document XML à traduire en requête REST"""

    def __init__(self, data, source):
        try:
            self.payload = xmltodict.parse(data)
        except xml.parsers.expat.ExpatError:
            raise SyntaxError('Incorrect XML format')
        else:
            self.xml = data

        if not self.is_valid:
            raise ValueError('Check the tags in your XML document')

        self.source = source
        self.version = 'v{}'.format(self.payload['root']['header']['version'])
        self.method = self.payload['root']['header']['param1'].lower()
        self.service = self.payload['root']['header']['param2']
        self.env = None
        if '/' in self.service:
            env, service = self.service.split('/')[-2:]
            if not env:
                self.env = None
            self.service = service
        self.data = self.payload['root'].get('data')
        if not self.data:
            self.data = {}
        self.username = self.payload['root']['header']['ident']
        self.token = self.data.get('token', None)

    @property
    def is_valid(self):
        """Checks validity of payload"""

        condition = (
            'root' in self.payload and
            'header' in self.payload['root'] and
            'version' in self.payload['root']['header'] and
            'param1' in self.payload['root']['header'] and
            'param2' in self.payload['root']['header'] and
            'ident' in self.payload['root']['header'] and
            'psw' in self.payload['root']['header'] and
            self.payload['root']['header']['param1'] in HTTP_VERBS
        )
        return condition

    @staticmethod
    def on_reception(ctx):
        soap_body = etree.tostring(ctx.in_body_doc)
        soap_body = xmltodict.parse(soap_body)

        # soap_body ressemblerait à ceci
        #   <ns0:BubbleRequest>
        #     <ns0:xml_in>
        #       <root>
        #         <header>
        #           <version>1</version>
        #           <param1>GET</param1>
        #           <param2>some_service</param2>
        #           <param3>/some/resource<param3>
        #           <ident>username_or_token</ident>
        #           <psw>password_if_necessary</psw>
        #         </header>
        #         <data>
        #           <token>token_if_allowed</token>
        #           <resource_params>
        #             <param1>variable_part_of_ressource_1</param1>
        #             <param2>variable_part_of_ressource_2</param2>
        #           </resource_params>
        #           <query_params>
        #             <arg_name>arg_value</arg_name>
        #           </query_params>
        #           <json_params>
        #             <key1>value1</key1>
        #             <key2>value2</key2>
        #           </json_params>
        #         </data>
        #       </root>
        #     </ns0:xml_in>
        #   </ns0:BubbleRequest>

        call_service_key = [
            key for key in soap_body.keys() if key.endswith('BubbleRequest')
        ]
        if len(call_service_key) != 1:
            raise BadRequestError()

        call_service_key = call_service_key.pop()
        payload_key = [
            key for key in soap_body[call_service_key].keys()
            if key.endswith('xml_in')
        ]
        if len(payload_key) != 1:
            raise BadRequestError()

    @staticmethod
    def response_rename_tag(master_key, iterable, result, index=None):
        """Rename a tag by pre-pending all keys up in hierarchy"""
        if isinstance(iterable, dict):
            for key, value in iterable.items():
                if index is not None:
                    new_key = '{}_{}:{}'.format(master_key, index, key)
                else:
                    new_key = '{}:{}'.format(master_key, key)
                SoapPayload.response_rename_tag(new_key, value, result)
        elif isinstance(iterable, list):
            for value in iterable:
                if isinstance(value, str):
                    new_key = '{}:{}'.format(master_key, iterable.index(value))
                    result.append((new_key, value))
                else:
                    SoapPayload.response_rename_tag(
                        master_key,
                        value,
                        result,
                        iterable.index(value)
                    )
        else:
            result.append((master_key, iterable))
        return result

    def translate(self):
        """Traduction du payload en requête REST.
        En fait on rappelle sponge, mais sur son interface
        REST cette fois.
        """
        api = Api(self.service, version=self.version, env=self.env)

        # récupération de l'adresse complète de la ressource
        resource = '{param3}/{params}'.format(
            param3=self.payload['root']['header'].get('param3', ''),
            params='/'.join(self.data.get('resource_params', {}).values())
        )
        resource = resource.rstrip('/')
        if not resource.startswith('/'):
            resource = '/' + resource

        # récupération des credentials
        password = self.payload['root']['header']['psw']
        username_or_token = self.username
        if self.token:
            password = ''
            username_or_token = self.token

        # récupération des paramètres in_URL
        params = self.data.get('query_params', None)

        # mise à jour du flag pour le formattage de la réponse plus tard
        if params and 'xml_rename_tags' in params:
            flag = params['xml_rename_tags']
            del params['xml_rename_tags']
        else:
            flag = False

        # récupération du document JSON à envoyer
        data = self.data.get('json_params', None)

        # on vérifie le token, ou bien on laisse passer la requête

        client_response = api.call(
            self.method,
            resource,
            headers={'X-Real-Ip': self.source},
            auth=(username_or_token, password),
            params=params,
            json=data
            )

        # format return
        result = {
            'root': {
                'header': {
                    'errorNum': 0,
                    'errorTx': '',
                },
                'data': {}
            }
        }
        content = client_response['content']
        headers = client_response['headers']
        _status = client_response['status']
        result['root']['data']['request_id'] = headers['X-Request-Id']
        if _status >= 400:
            if not content['code'].startswith('04-'):
                result['root']['data']['code'] = content['code']
                result['root']['data']['message'] = content['message']
            else:
                result['root']['header']['errorNum'] = _status
                result['root']['header']['errorTx'] = content['message']
        else:
            if content and flag:
                response = []
                for key in content:
                    response = SoapPayload.response_rename_tag(
                        key,
                        content[key],
                        response
                    )
                new_data = {}
                temp = []
                response = sorted(response)
                for i, j in response:
                    bar = []
                    keys = i.split(':')
                    for k in range(1, len(keys) + 1):
                        bar.append('_'.join(keys[0:k]))
                    temp.append((bar, j))
                for path, value in temp:
                    current_level = new_data
                    for index, part in enumerate(path):
                        if part not in current_level:
                            if index < len(path) - 1:
                                current_level[part] = {}
                            else:
                                current_level[part] = value
                        current_level = current_level[part]
                new_data = xmltodict.unparse({'data': new_data})
                new_data = new_data.split('\n')[1]
                if '<token>' not in new_data:
                    regex = r"_\d{1,}"
                    new_data = re.sub(regex, '', new_data, 0)
                new_data = xmltodict.parse(new_data)
                result['root']['data'].update(new_data['data'])
            else:
                result['root']['data'].update(content)
        return result
# EOF
