# -*- coding: utf-8 -*-

from flask import jsonify, request

from blueflask.lib.infos import success
from blueflask.lib.errors import not_found

from ....consts import REQUESTS_LOGS
from ....consts import SERVICE_CODE
from . import api

import datetime
import json


@api.route('/requests', methods=['POST'])
def record_request_response():
    """
    Enregistrement du résultat d'une requête
    ---
    tags:
      - requests
    """
    data = request.get_json()
    request_id = data['request_id']
    service = data['service']
    index = data.get('index')
    if index:
        del data['index']
    del data['request_id']
    del data['service']
    key = 'request:{service}:{request_id}:{timestamp}'.format(
        service=service,
        request_id=request_id,
        timestamp=datetime.datetime.now().strftime('%d%m%Y%H%M%S')
        )
    if index:
        key = '{key}:{index}'.format(key=key, index=index)
    REQUESTS_LOGS.hmset(key, data)
    REQUESTS_LOGS.expire(key, 2678400)
    return success(service_code=SERVICE_CODE)


@api.route('/requests/<string:service>/<string:request_id>', methods=['GET'])
def get_request_response(service, request_id):
    """
    Résultat d'une requête.
    Cette ressource renvoie la réponse correspondant à la requête
    identifée par le request-id passé dans l'URL
    ---
    tags:
      - requests
    responses:
      200:
        description: réponse à la requête
      404:
        description: ce request-id n'existe pas
    """
    keys = REQUESTS_LOGS.keys('request:{}:{}:*'.format(service, request_id))
    if not keys:
        return not_found(service_code=SERVICE_CODE)
    else:
        result = {
            'code': '04-200',
            'info': 'Résultats de la requête {}'.format(request_id),
            'status': 200,
            'message': []
            }
        statuses = {}
        for key in keys:
            response = REQUESTS_LOGS.hgetall(key)
            response = json.loads(response)
            infos = {
                key.decode('utf-8'): value.decode('utf-8')
                for key, value in response.items()
                }
            code = infos['code']
            if code not in statuses:
                statuses[code] = [infos]
            else:
                statuses[code].append(infos)
        for code, data in statuses.items():
            result['message'].append(infos)
        return jsonify(result)

# EOF
