# -*- coding: utf-8 -*-

from flask import jsonify, request

from blueflask.lib.infos import HTTPInfo
from blueflask.lib.errors import HTTPError
from blueflask.lib.errors import not_found, unauthorized

from ....consts import SERVICE_CODE
from .... import gateway
from . import api


@api.route('/<string:env>/auth/<string:path>', methods=['POST'])
@api.route('/auth/<string:path>', methods=['POST'])
@api.route('/auth', methods=['POST'])
def ldap_auth(path=None, env=None):
    auth = request.authorization
    password = auth.password
    username = auth.username
    params = request.args or {}
    data = request.form or {}
    json_doc = request.get_json() or {}
    try:
        gw = gateway.Api('ldap', env=env)
    except (KeyError, ValueError):
        return not_found(service_code=SERVICE_CODE)
    else:
        if not path:
            uri = '/auth'
        else:
            uri = '/auth/{}'.format(path)
        result = gw.call(
            request.method,
            uri,
            headers=request.headers,
            auth=(username, password),
            params=params,
            data=data,
            json=json_doc,
            verify=False
            )
        status = result['status']
        content = result['content']
        if isinstance(result['content'], str):
            if status >= 400:
                response = HTTPError(
                    status,
                    service_code=SERVICE_CODE
                    ).response(content)
            else:
                response = HTTPInfo(
                    status,
                    service_code=SERVICE_CODE
                    ).response(content)
        else:
            response = jsonify(content)
            response.status_code = status
        return response


@api.route(
    '/<string:env>/<string:service>/v<int:version>/<path:path>',
    methods=['GET', 'POST', 'PUT', 'DELETE']
    )
@api.route(
    '/<string:service>/v<int:version>/<path:path>',
    methods=['GET', 'POST', 'PUT', 'DELETE']
    )
def api_gateway(service, version, path, env=None):
    """
    API Gateway.
    ---
    tags:
      - Gateway
    """
    auth = request.authorization
    if not auth:
        # on n'accepte aucune requête qui ne comporte
        # pas de paramètres d'authentification
        # HTTPBasicAuth
        return unauthorized(code='04-401')

    password = auth.password
    username = auth.username
    params = request.args or {}
    data = request.form or {}
    json_doc = request.get_json() or {}

    try:
        gw = gateway.Api(service, version=version, env=env)
    except (KeyError, ValueError):
        return not_found(service_code=SERVICE_CODE)
    else:
        result = gw.call(
            request.method,
            '/{}'.format(path),
            headers=request.headers,
            auth=(username, password),
            params=params,
            data=data,
            json=json_doc
            )
        status = result['status']
        content = result['content']
        if isinstance(result['content'], str):
            if status >= 400:
                response = HTTPError(
                    status,
                    service_code=SERVICE_CODE
                    ).response(content)
            else:
                response = HTTPInfo(
                    status,
                    service_code=SERVICE_CODE
                    ).response(content)
        else:
            response = jsonify(content)
            response.status_code = status
        return response

# EOF
