#!/usr/bin/env python
# encoding: utf-8

"""
Gestion des erreurs de l'API
"""

from flask import jsonify, request

from blueflask.lib.errors import unauthorized, bad_request
from blueflask.lib.errors import HTTPError

import functools

ERRORS = {
    '040-05-4000': (400, 'paramètres incomplets'),
    '040-05-4001': (400, 'souci non défini'),
    '040-05-4003': (400, 'un voucher bip comporte 15 chiffres'),
    '040-05-4004': (400, 'carte inexistante'),
    '040-05-4016': (400, 'aucune recharge en cours'),
    '040-05-4017': (400, 'action non autorisée'),
    '040-05-4018': (400, 'un soucis sur le device encours'),
    '040-05-4019': (400, 'aucune offre disponible pour ce device'),
    '040-05-4011': (401, 'l\'identifiant n\'existe pas'),
    '040-05-4012': (401, 'code secret non défini'),
    '040-05-4013': (401, 'code secret incorrect'),
    '040-05-4014': (401, 'solde insuffisant'),
    '040-05-4015': (401, 'aucun abonnement disponible,'),
    '040-05-5001': (500, 'erreur réseau interne'),
    '040-05-5002': (500, 'timeout')
    }


def error(code):
    return HTTPError(ERRORS[code][0]).response(
        message=ERRORS[code][1],
        code=code
        )


def error_handling(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        data = request.get_json()
        mandatories = {'operator', 'service_type'}
        keys = set(data.keys())
        if not mandatories.intersection(keys):
            return error('040-05-4001')
        else:
            result = func(*args, **kwargs)
            if not isinstance(result, dict):  # result['status'] == 200
                return result
            else:
                # on a remonté les erreurs du backend telles quelles
                # du coup on fait de la réécriture
                code = result['data']['code']
                if result['status'] == 400:
                    if code.startswith('042-01'):  # erreurs bluebase-services
                        codes = [
                            '702',
                            '703',
                            '704',
                            '707',
                            '708',
                            '709',
                            '713'
                            ]
                        if code.split('-')[-1] in codes:
                            return error('040-05-4001')
                        elif code == '042-01--1':
                            return error('040-05-4001')
                        elif code.split('-')[-1] in ['705', '706']:
                            return error('040-05-4011')
                        elif code.split('-')[-1] in ['712']:
                            return error('040-05-4018')
                        elif code.split('-')[-1] in ['201']:
                            return error('040-05-4017')
                        elif code.split('-')[-1] in ['717', '710']:
                            return error('040-05-4014')
                        elif code.split('-')[-1] in ['721']:
                            return error('040-05-4016')
                        elif code.split('-')[-1] in ['722']:
                            return error('040-05-4015')
                        elif code.split('-')[-1] in ['723']:
                            return error('040-05-4019')
                        else:
                            return bad_request()
                    elif code.startswith('040-02'):    # erreurs OCS
                        if code == '040-02-4003':
                            return error('040-05-4003')
                        elif code == '040-02-4004':
                            return error('040-05-4004')
                        else:
                            return error('040-05-4001')
                elif result['status'] == 401:
                    if code in ['040-02-401', '042-01-401', '044-01-401']:
                        return unauthorized()
                    elif code == '040-02-4010':
                        return error('040-05-4013')
                elif result['status'] == 404:
                    if code.startswith('040-02'):  # erreurs OCS
                        if code == '040-02-4041':
                            return error('040-05-4001')
                        else:
                            return error('040-05-4012')
                elif result['status'] == 503:
                    return error('040-05-5001')
                elif result['status'] == 504:
                    return error('040-05-5002')
                elif result['status'] == 500:
                    response = jsonify(result['data'])
                    response.status_code = 500
                    return response
    return wrapper
# EOF
