#!/usr/bin/env python
# encoding: utf-8

from flask import jsonify, request

from blueflask.lib.errors import not_allowed, bad_request
from blueflask.lib.infos import success

from ...bluebase import Bluebase
from ...ocs import OCS
from ..errors import error_handling
from . import api


BLUEBASE_CLIENT = Bluebase()
OCS_CLIENT = OCS()


@api.route('/clients/authenticate', methods=['POST'])
@error_handling
def authenticate():
    """
    Récupération des informations d'un client Blueline \
    à partir de son identifiant.
    En fournissant un identifiant (et en le caractérisant), \
    il est possible de demander à Blueline de retourner les informations \
    de base du client ainsi recherché.
    ---
    tags:
      - Authentication
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: number
              description: Status HTTP de la requête
            code:
              type: string
              description: Code d'erreur interne aux APIs de Blueline
            error:
              type: string
              description: Message court décrivant l'erreur
            message:
              type: string
              description: Message d'erreur
    parameters:
      - name: operator
        in: body
        required: true
        description: Nom de l'opérateur réseau
        schema:
          type: string
          enum:
            - bip
            - airtel
            - orange
            - telma
      - name: caller_num
        in: body
        type: string
        required: true
        description: Numéro de téléphone de l'utilisateur
      - name: service_type
        in: body
        required: true
        description: Type de service
        schema:
          type: string
          enum:
            - tv
            - internet
            - bip
      - name: login_type
        in: body
        required: true
        description: Type d'authentification
        schema:
          type: string
          enum:
            - tv_card_num
            - internet_num
            - mail_auth
            - client_refnum
            - bip_num
      - name: login
        in: body
        type: string
        required: true
        description: Identifiant
    responses:
      200:
        description: Informations sur le client Blueline
        properties:
          customer_id:
            type: string
            description: Identifiant unique du client Blueline
          name:
            type: string
            description: Prénom du client Blueline
          last_name:
            type: string
            description: Nom du client Blueline
          device_name:
            type: array
            items:
              type: string
            description: Liste des produits du client
      401:
        description: Echec authentification
        $ref: "#/definitions/Error"
      400:
        description: Erreur sur un des paramètres en entrée
        $ref: "#/definitions/Error"
      405:
        description: Action non autorisée
        $ref: "#/definitions/Error"
      500:
        description: Une erreur interne est survenue
        $ref: "#/definitions/Error"
    """
    data = request.get_json()
    auth = request.authorization
    if data['service_type'] == 'bip':
        return not_allowed(
            message='action non autorisée',
            code='040-05-405'
            )
    else:
        data['request'] = request
        try:
            result = BLUEBASE_CLIENT.authentication(auth.username, **data)
        except SyntaxError:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
                )
        else:
            if result['status'] == 200:
                if data['caller_num'].startswith('03900'):
                    result['data']['num_is_mine'] = 'Vrai'
                response = {
                    'customer_id': result['data']['customer_id'],
                    'device_name': result['data'].get('device_name', []),
                    'last_name': result['data']['last_name'],
                    'name': result['data']['name'],
                    'num_is_mine': result['data']['num_is_mine']
                    }
                return jsonify(response)
            else:
                return result


@api.route('/clients/authenticate/has-code', methods=['POST'])
@error_handling
def has_code():
    """
    Vérifier si un utilisateur Bip a un code secret.
    ---
    tags:
      - Authentication
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: operator
        in: body
        required: true
        description: Identifiant de l'opérateur
        schema:
          type: string
          enum:
            - telma
            - orange
            - airtel
            - bip
      - name: caller_num
        in: body
        type: string
        description: Numéro de téléphone de l'utilisateur du menu
      - name: service_type
        in: body
        required: true
        description: Type de service
        schema:
          type: string
          enum:
            - tv
            - internet
            - bip
    """
    data = request.get_json()
    auth = request.authorization
    data['request'] = request
    if data['operator'] != 'bip':
        return not_allowed(
            message=(
                "api seulement accessible via bip"
                ),
            code='040-05'
            )
    else:
        try:
            result = OCS_CLIENT.has_code(auth.username, **data)
        except SyntaxError as exc:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
                )
        else:
            if result['status'] == 200:
                return success()
            else:
                return result


@api.route('/clients/authenticate/verify-code', methods=['POST'])
@error_handling
def verify_code():
    """
    Vérifier si le code secret passé en paramètre existe dans la base.
    ---
    tags:
      - Authentication
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: operator
        in: body
        required: true
        description: Identifiant de l'opérateur
        schema:
          type: string
          enum:
            - telma
            - orange
            - airtel
            - bip
      - name: caller_num
        in: body
        type: string
        description: Numéro de téléphone de l'utilisateur du menu
      - name: service_type
        in: body
        required: true
        description: Type de service
        schema:
          type: string
          enum:
            - tv
            - internet
            - bip
      - name: secret_code
        in: body
        type: string
        required: true
        description: Code secret
    """
    data = request.get_json()
    auth = request.authorization
    data['request'] = request
    if data['operator'] != 'bip':
        return not_allowed("api seulement accessible via bip")
    else:
        try:
            result = OCS_CLIENT.verify_code(auth.username, **data)
        except SyntaxError as exc:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
                )
        else:
            if result['status'] == 200:
                return success()
            else:
                return result


@api.route('/clients/authenticate/create-code', methods=['POST'])
@error_handling
def create_code():
    """
     Initialiser le code secret d'un utilisateur du réseau Bip.
    ---
    tags:
      - Authentication
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: operator
        in: body
        required: true
        description: Identifiant de l'opérateur
        schema:
          type: string
          enum:
            - telma
            - orange
            - airtel
            - bip
      - name: caller_num
        in: body
        type: string
        description: Numéro de téléphone de l'utilisateur du menu
      - name: service_type
        in: body
        required: true
        description: Type de service
        schema:
          type: string
          enum:
            - tv
            - internet
            - bip
      - name: secret_code
        in: body
        type: string
        required: true
        description: nouveau code secret
    """
    data = request.get_json()
    auth = request.authorization
    data['request'] = request
    if data['operator'] != 'bip':
        return not_allowed(message="api seulement accessible via bip")
    else:
        try:
            result = OCS_CLIENT.set_code(auth.username, **data)
        except SyntaxError as exc:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
                )
        else:
            if result['status'] == 200:
                return success()
            else:
                return result


@api.route('/clients/authenticate/reset-code', methods=['POST'])
@error_handling
def reset_code():
    """
    Changer le code secret d'un utilisateur du réseau Bip.
    ---
    tags:
      - Authentication
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: operator
        in: body
        required: true
        description: Identifiant de l'opérateur
        schema:
          type: string
          enum:
            - telma
            - orange
            - airtel
            - bip
      - name: caller_num
        in: body
        type: string
        description: Numéro de téléphone de l'utilisateur du menu
      - name: service_type
        in: body
        required: true
        description: Type de service
        schema:
          type: string
          enum:
            - tv
            - internet
            - bip
      - name: old_secret_code
        in: body
        type: string
        required: true
        description: l'ancien code secret de l'utilisateur
      - name: new_secret_code
        in: body
        type: string
        required: true
        description: le nouveau code secret de l'utilisateur

    """
    data = request.get_json()
    auth = request.authorization
    data['request'] = request
    data['secret_code'] = data['old_secret_code']
    del data['old_secret_code']
    if data['operator'] != 'bip':
        return not_allowed("api seulement accessible via bip")
    else:
        try:
            result = OCS_CLIENT.verify_code(auth.username, **data)
        except SyntaxError as exc:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
                )
        else:
            if result['status'] == 200:
                data['secret_code'] = data['new_secret_code']
                del data['new_secret_code']
                response = OCS_CLIENT.set_code(auth.username, **data)
                if response['status'] == 200:
                    return success()
                else:
                    return response
            else:
                return result


@api.route(
    '/clients/<string:customer_id>/subscriptions',
    methods=['GET']
    )
@error_handling
def get_subscription_infos(customer_id):
    """
    Récupération des infos d'un abonnement client.
    Cette ressource permet de récupérer les infos liées à un abonnement.
    ---
    tags:
      - Subscriptions
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: operator
        in: body
        required: true
        description: Identifiant de l'opérateur
        schema:
          type: string
          enum:
            - telma
            - orange
            - airtel
            - bip
      - name: caller_num
        in: body
        type: string
        description: Numéro de téléphone de l'utilisateur du menu
      - name: service_type
        in: body
        required: true
        description: Type de service
        schema:
          type: string
          enum:
            - tv
            - internet
            - bip
      - name: device_name
        in: body
        type: string
        description: Identifiant de l'abonnement, utile pour services hors Bip
    responses:
      200:
        description: détails sur un abonnement
        properties:
          customer_id:
            type: string
            description: Identifiant unique du client
          subscriptions:
            description: Liste des packages associés à cet abonnement
            type: array
            items:
              properties:
                name:
                  description: Nom du package
                  type: string
                value:
                  type: string
                  description: Date de validité ou Solde
      401:
        description: échec de l'authentification
        $ref: "#/definitions/Error"
      400:
        description: les paramètres en entrées sont incorrects
        $ref: "#/definitions/Error"
      500:
        description: une erreur interne est survenue
        $ref: "#/definitions/Error"
    """
    data = request.get_json()
    auth = request.authorization
    data['request'] = request
    data['customer_id'] = customer_id
    if data['service_type'] == 'bip':
        if data['caller_num'] != data['customer_id']:
            return not_allowed(
                message='action non autorisée',
                code='040-05-405'
                )
        else:
            try:
                result = OCS_CLIENT.get_infos(auth.username, **data)
                if result['status'] in [200, 204]:
                    response = {
                        'customer_id': customer_id,
                        'subscriptions': result['data'],
                        'length': len(result['data']),
                    }
                    return jsonify(response)
                else:
                    return result

            except SyntaxError:
                return bad_request(
                    message="paramètres incomplets",
                    code='040-05-400'
                    )
    else:
        try:
            result = BLUEBASE_CLIENT.subscriptions(auth.username, **data)
            if result['status'] in [200, 204]:
                if result['num_is_mine'] == 'Vrai' or \
                   data['caller_num'].startswith('03900') or \
                   data['caller_num'].startswith('3900'):
                    response = {
                        'customer_id': customer_id,
                        'subscriptions': result['data'],
                        'length': len(result['data']),
                    }
                    return jsonify(response)
                else:
                    return not_allowed(
                        message="action non autorisée",
                        code='040-05-405'
                    )
            else:
                return result
        except SyntaxError:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
                )


@api.route('/clients/<string:customer_id>/offers', methods=['GET'])
@error_handling
def get_available_offers(customer_id):
    """
    Permet de récupérer les offres disponibles à l'achat.
    En fonction du type de service et le solde actuel du numéro \
    en train d'utiliser le menu USSD, on retourne les offres disponibles \
    à l'achat.
    ---
    tags:
      - Offers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: operator
        in: bodyc
        required: true
        description: Identifiant de l'opérateur
        schema:
          type: string
          enum:
            - telma
            - orange
            - airtel
            - bip
      - name: caller_num
        in: body
        type: string
        description: Numéro de téléphone de l'utilisateur du menu, si hors Bip
      - name: service_type
        in: body
        required: true
        description: Type de service
        schema:
          type: string
          enum:
            - tv
            - internet
            - bip
      - name: device_name
        in: body
        required: True
        type: string
        description: Identifiant de l'abonnement
      - name: balance
        in: body
        type: string
        description: Solde du caller_num si l'utilisateur est extérieur à Bip
    responses:
      200:
        description: liste des offres
        properties:
          length:
            type: number
            description: Nombre d'offres disponibles à l'achat
          service_type:
            type: string
            description: Type du service
          offers:
            type: array
            description: Liste des offres
            items:
              properties:
                name:
                  type: string
                  description: Nom de l'offre
                price:
                  type: number
                  description: Prix de l'offre
                offer_id:
                  type: string
                  description: Identifiant de l'offre
      401:
        description: échec de l'authentification
        $ref: "#/definitions/Error"
      400:
        description: les paramètres en entrées sont incorrects
        $ref: "#/definitions/Error"
      500:
        description: une erreur interne est survenue
        $ref: "#/definitions/Error"
    """
    data = request.get_json()
    auth = request.authorization
    data['request'] = request
    data['customer_id'] = customer_id

    if data['operator'] == 'bip':  # on doit chercher le solde de l'utilisateur
        try:
            result = OCS_CLIENT.check_balance(auth.username, **data)
        except SyntaxError:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
                )
        else:
            if result['status'] in [200, 204]:
                data['balance'] = result['data']['balance']
            else:
                return result
    else:
        pass
        # si autre operateur on ne passe pas par ocs

    if data['service_type'] == 'bip':
        if data['caller_num'] != data['customer_id']:
            return not_allowed(
                message='action non autorisée',
                code='040-05-405'
                )
        else:
            try:
                result = OCS_CLIENT.get_offers(auth.username, **data)
            except SyntaxError:
                return bad_request(
                    message="paramètres incomplets",
                    code='040-05-400'
                    )
    else:
        try:
            result = BLUEBASE_CLIENT.offers(auth.username, **data)
        except SyntaxError:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
                )

    if result['status'] in [200, 204]:
        response = {
            'service_type': data['service_type'],
            'offers': result['data'],
            'length': len(result['data'])
            }
        return jsonify(response)
    else:
        return result


@api.route('/clients/<string:customer_id>/offers', methods=['POST'])
@error_handling
def activate_offer(customer_id):
    """
    Activation d'une offre.
    ---
    tags:
      - Offers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: operator
        in: body
        required: true
        description: Identifiant de l'opérateur
        schema:
          type: string
          enum:
            - telma
            - orange
            - airtel
            - bip
      - name: caller_num
        in: body
        required: True
        type: string
        description: Numéro de téléphone de l'utilisateur du menu
      - name: service_type
        in: body
        required: true
        description: Type de service
        schema:
          type: string
          enum:
            - tv
            - internet
            - bip
      - name: device_name
        in: body
        required: True
        type: string
        description: Identifiant de l'abonnement
      - name: offer_name
        in: body
        description: Type d'offre bip
        schema:
          type: string
          enum:
            - voucher
            - offer
            - transfer
      - name: offer_ref
        in: body
        type: string
        required: true
        description: nom offre tv ou internet
      - name: offer_id
        in: body
        type: string
        required: true
        description: Identifiant de l'offre
      - name: balance
        in: body
        type: string
        description: Solde du caller_num si l'utilisateur est extérieur à Bip
    responses:
      204:
        description: L'achat de l'offre a réussi
      401:
        description: échec de l'authentification
        $ref: "#/definitions/Error"
      400:
        description: les paramètres en entrées sont incorrects
        $ref: "#/definitions/Error"
      500:
        description: une erreur interne est survenue
        $ref: "#/definitions/Error"
    """
    data = request.get_json()
    auth = request.authorization
    data['request'] = request
    data['customer_id'] = customer_id
    if data['service_type'] == 'bip':
        if data['offer_name'] == 'voucher':
            try:
                result = OCS_CLIENT.buy_voucher(auth.username, **data)
                if result['status'] == 200:
                    response = {
                        'face_value': result['data']['face_value']
                        }
                    return jsonify(response)
                else:
                    return result
            except SyntaxError:
                return bad_request(
                    message="paramètres incomplets",
                    code='040-05-400'
                    )
        elif data['offer_name'] == 'transfer':
            try:
                result = OCS_CLIENT.transfer(auth.username, **data)
            except SyntaxError:
                return bad_request(
                    message="paramètres incomplets",
                    code='040-05-400'
                    )
        elif data['offer_name'] == 'offer':
            try:
                result = OCS_CLIENT.buy_offer(auth.username, **data)
            except SyntaxError:
                return bad_request(
                    message="paramètres incomplets",
                    code='040-05-400'
                    )
    else:
        if data['operator'] == 'bip':
            try:
                result = BLUEBASE_CLIENT.buy_recharge(auth.username, **data)
                result['token'] = auth.username
                return jsonify(result)
            except SyntaxError:
                return bad_request(
                    message="paramètres incomplets",
                    code='040-05-400')
        # au cas où c'est un partenaire qui fait la requête
        else:
            data['balance'] = data.get('balance', 0)
        try:
            result = BLUEBASE_CLIENT.buy_recharge(auth.username, **data)
        except SyntaxError:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
                )
    if result['status'] in [200, 204]:
        return success()
    else:
        return result


@api.route('/clients/<string:phone_number>/locale', methods=['GET'])
def locale(phone_number):
    """
    Récupération de la langue utilisée par le client bip.
    ---
    tags:
      - Locale
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: operator
        in: body
        required: true
        description: Identifiant de l'opérateur
        schema:
          type: string
          enum:
            - telma
            - orange
            - airtel
            - bip
      - name: caller_num
        in: body
        required: True
        type: string
        description: Numéro de téléphone de l'utilisateur du menu
      - name: service_type
        in: body
        required: true
        description: Type de service
        schema:
          type: string
          enum:
            - tv
            - internet
            - bip
     responses:
      204:
        description: récupération de la langue réussie
      401:
        description: echec de la requ^êtz
        $ref: "#/definitions/Error"
      400:
        description: les paramètres en entrées sont incorrects
        $ref: "#/definitions/Error"
      500:
        description: une erreur interne est survenue
        $ref: "#/definitions/Error"
    """

    data = request.get_json()
    auth = request.authorization
    if (data['service_type'] == 'bip' and
            data['operator'] == 'bip'):

        try:
            result = OCS_CLIENT.get_language(auth.username, **data)
        except SyntaxError:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
                )
        return result
    else:
        return bad_request(
            message="action non autorisee",
            code='040-05-400'
            )
