#!/usr/bin/env python
# encoding: utf-8

import rest_client

from . import CONFIG
from .core import check_parameters
SERVICE_CODE = '040-05'


class OCS(object):
    """
    une classe pour les requêtes à destination de l'api OCS
    """

    def __init__(self):
        self.url = CONFIG['SERVICES']['ocs']
        self.client = rest_client.RestClient(self.url)

    @check_parameters('caller_num')
    def get_language(self, token, **data):
        request = data['request']
        caller_num = data['caller_num']
        del data['request']
        response = self.client.get(
                '/subscribers/{caller_num}'.format(
                    caller_num=caller_num),
                auth=(token, ''),
                headers=request.headers,
                params={
                    'key': 'language_code'
                    }
                )
        return {
            'status': response['status'],
            'data': response['content']['message']
            }

    @check_parameters('caller_num', 'language_code')
    def set_language(self, token, **data):
        request = data['request']
        caller_num = data['caller_num']
        del data['request']
        response = self.client.put(
            '/subscribers/{caller_num}'.format(
                caller_num=caller_num),
            auth=(token, ''),
            headers=request.headers,
            json={
                'language_code': data['language_code']
                }
            )
        return {
            'status': response['status'],
            'data': response['content']['message']
            }

    @check_parameters('caller_num')
    def has_code(self, token, **data):
        request = data['request']
        del data['request']
        response = self.client.get(
            '/auth/{}'.format(data['caller_num']),
            auth=(token, ''),
            headers=request.headers
            )
        return {
            'status': response['status'],
            'data': response['content']['message']
            }

    @check_parameters('caller_num', 'secret_code')
    def verify_code(self, token, **data):
        request = data['request']
        del data['request']
        response = self.client.post(
            '/auth/verify-code',
            auth=(token, ''),
            headers=request.headers,
            json={
                'phonenumber': data['caller_num'],
                'code': data['secret_code']
                }
            )
        return {
            'status': response['status'],
            'data': response['content']['message']
            }

    @check_parameters('caller_num', 'secret_code')
    def set_code(self, token, **data):
        request = data['request']
        del data['request']
        response = self.client.post(
            '/auth',
            auth=(token, ''),
            headers=request.headers,
            json={
                'phonenumber': data['caller_num'],
                'code': data['secret_code']
                }
            )
        return {
            'status': response['status'],
            'data': response['content']['message']
            }

    @check_parameters('customer_id')
    def get_infos(self, token, **data):
        """
        On reçoit de l'API OCS un document comme ceci:
        {
            'phonenumber': 0390390169,
            'length': 2;
            'infos': [
                {
                'name': 'principal',
                'balance': '200',
                },
                {
                'name': 'Tsik machin',
                'balance': '0s, 40MO, 4sms',
                },
            ]
        }
        """
        request = data['request']
        phonenumber = data['customer_id']
        del data['customer_id']
        del data['request']
        response = self.client.get(
            '/subscribers/{phonenumber}'.format(
                phonenumber=phonenumber
                ),
            auth=(token, ''),
            headers=request.headers
            )
        if response['status'] == 200:
            infos = [
                {
                    'name': info['name'],
                    'value': info['balance']
                    }
                for info in response['content']['message']['subscriptions']
                ]
            result = {
                'status': response['status'],
                'data': infos
                }
            return result
        elif response['status'] == 204:
            infos = []
            result = {
                'status': response['status'],
                'data': infos
                }
            return result
        else:
            return {
                'status': response['status'],
                'data': response['content']['message']
                }

    @check_parameters('customer_id', 'balance')
    def get_offers(self, token, **data):
        response = self.client.get(
            '/services/offers',
            auth=(token, '')
            )
        if response['status'] == 200:
            infos = [
                offer for offer in response['content']['message']['offers']
                if float(offer['price']) <= float(data['balance'])
                ]
        else:
            infos = response['content']['message']
        result = {
            'status': response['status'],
            'data': infos
            }
        return result

    @check_parameters('offer_id', 'customer_id')
    def buy_voucher(self, token, **data):
        phonenumber = data['customer_id']
        # buy voucher
        request = data['request']
        del data['request']
        code = data['offer_id']
        response = self.client.post(
            '/services/vouchers',
            auth=(token, ''),
            headers=request.headers,
            json={'phonenumber': phonenumber, 'code': code}
        )
        return {
            'status': response['status'],
            'data': response['content']['message']
            }

    @check_parameters('offer_id', 'customer_id')
    def buy_offer(self, token, **data):
        phonenumber = data['customer_id']
        # buy voucher
        request = data['request']
        del data['request']
        offer_id = data['offer_id']
        response = self.client.post(
            '/services/offers',
            auth=(token, ''),
            headers=request.headers,
            json={'phonenumber': phonenumber, 'offer_id': offer_id}
        )
        return {
            'status': response['status'],
            'data': response['content']['message']
            }

    @check_parameters('customer_id', 'device_name', 'amount')
    def transfer(self, token, **data):
        request = data['request']
        del data['request']
        receiver = data['device_name']
        sender = data['customer_id']
        amount = data['amount']
        response = self.client.post(
            '/services/transfers',
            auth=(token, ''),
            headers=request.headers,
            json={'sender': sender, 'receiver': receiver, 'amount': amount}
        )
        return {
            'status': response['status'],
            'data': response['content']['message']
            }

    @check_parameters('customer_id')
    def check_balance(self, token, **data):
        """
        Récupération du solde téléphonique (2000) d'un abonné
        """
        request = data['request']
        phonenumber = data['caller_num']
        del data['service_type']
        del data['customer_id']
        del data['request']
        response = self.client.get(
            '/subscribers/{phonenumber}/balances'.format(
                phonenumber=phonenumber
                ),
            auth=(token, ''),
            headers=request.headers,
            json={'account_type': '2000'}
            )
        return {
            'status': response['status'],
            'data': response['content']['message']
            }

    @check_parameters('customer_id', 'amount')
    def update_balance(self, token, **data):
        """
        Modification du solde téléphonique (2000) d'un abonné
        """
        response = self.client.put(
            '/subscribers/{phonenumber}/balances'.format(
                phonenumber=data['caller_num']
                ),
            auth=(token, ''),
            headers=data['request'].headers,
            json={
                'amount': data['amount'],
                'operation_type': '2',
                'account_type': 2000
                }
            )
        return {
            'status': response['status'],
            'data': response['content']['message']
            }
