#!/usr/bin/env python
# encoding: utf-8

import json
import threading

import datetime
import pika
import rest_client
from blueflask.lib.errors import bad_request

from .ocs import OCS
from .core import write_bip_transaction_to_db

from . import LOGS
from . import CONFIG
from . import REQUEST_DATABASE
from .locale import LOCALE_MSG

SERVICE_CODE = "040-05"
OCS_OP_TYPE = CONFIG["APP"]["ocs_operation_type"]
OCS_ACC_TYPE = CONFIG["APP"]["ocs_account_type"]
PARTNER_CREDIT_LIMIT = CONFIG["MISC"]["partner_credit_limit"]
# traitement des msg venant de rabbitmq


def process(queue):
    """ Traitement des files d'attentes """
    connection = pika.BlockingConnection(
        pika.ConnectionParameters(host="localhost", virtual_host="/ussd")
    )

    channel = connection.channel()
    channel.basic_consume(Callback.call, queue=queue)
    channel.start_consuming()


class Worker(threading.Thread):
    def __init__(self, name, arg):
        threading.Thread.__init__(self)
        self.name = name
        self.arg = arg

    def run(self):
        process(self.arg)


class Callback(object):
    def __init__(self, method, **kwargs):
        self.method = method
        self.queue = "activate.ussd"
        self.channel = kwargs.get("channel")
        self.url_blue = CONFIG["SERVICES"]["bluebase"]
        self.url_smsc = CONFIG["SERVICES"]["smsc"]
        self.url_ocs = CONFIG["SERVICES"]["ocs"]
        self.client_blue = rest_client.RestClient(self.url_blue)
        self.client_smsc = rest_client.RestClient(self.url_smsc)
        self.client_ocs = rest_client.RestClient(self.url_ocs)

    @staticmethod
    def call(ch, method, headers, body):
        LOGS.logger.info("Key error content debug".format(body))
        params = body.decode("utf-8")
        document = json.loads(params)
        # on aura besoin des paramètres suivants
        # pour l'activation
        # et l'envoi d'un sms
        params = document["data"]
        LOGS.logger.info("Input: {}".format(params))
        callback = Callback(method, channel=ch)
        # ici on demande l'activation d'une offre
        # la methode à appeler dépendra de l'operateur
        # pareil pour la notification
        # si c'est bip
        # on envoi directement un sms
        # sinon
        # on enregistre le résultat
        # sur redis
        if params["operator"] == "bip":
            state = callback.activate_offer_bip(**params)
            params["offer_state"] = state
            callback.notify_via_sms(**params)

            params["offer_name"] = ""
            callback.write_to_redis(**params)

        else:
            state = callback.activate_offer_partner(**params)
            params["offer_state"] = state
            callback.write_to_redis(**params)
        LOGS.logger.info(state)
        LOGS.logger.info(params["offer_state"])
        ch.basic_ack(delivery_tag=method.delivery_tag)

    # transfert de credit via partenaire
    # utiliser payment au lieu de adjust_account
    def recharging_bip_account(self, **params):
        token = params["token"]
        data_headers = params["headers"]
        LOGS.logger.info(params)
        amount = int(params["amount"])
        add_infos = "CreditFrom+\{}+.".format(params["caller_num"])
        # autoriser le transfert de credit
        # superieur ou egale à limite_credit Ar
        # autoriser le transfert de credit
        # superieur ou egale à limite_credit Ar
        if amount >= int(PARTNER_CREDIT_LIMIT):
            response = self.client_ocs.post(
                "/services/transfers",
                auth=(token, ""),
                headers=data_headers,
                json={
                    "receiver": params["device_id"],
                    "sender": params["caller_num"],
                    "amount": params["amount"],
                    "infos": add_infos,
                },
            )
        else:
            response = (
                {
                    "status": 400,
                    "content": "montant insuffisant",
                    "info": "",
                    "code": "040-05-4031",
                },
            )
        LOGS.logger.info("response from OCS: {}".format(response))
        return response

    def update_customer_balance(self, data_headers, **params):
        amount = int(params["amount"])
        token = params["token"]
        phonenumber = params["device_id"]
        bundle = params["bundle_name"].replace(" ", "")
        add_infos = "{}+\.".format(bundle)
        response = self.client_ocs.put(
            "/subscribers/{phonenumber}/balances".format(phonenumber=phonenumber),
            auth=(token, ""),
            headers=data_headers,
            json={
                "amount": amount,
                "operation_type": "2",
                "account_type": 2000,
                "infos": add_infos,
            },
        )
        return response

    def buying_bip_bundle(self, **params):
        """
           achat d'un bundle bip via un compte  partenaire
           recharger d'abord le compte bip du montant du bundle
           ensuite effectuer l'achat
           si il y a echec, il faudra faire un rollback
        """
        data = params
        phonenumber = params["device_id"]
        data_headers = params["headers"]
        offer_id = params["offer_id"]
        amount = params["amount"]
        token = params["token"]
        try:
            bip_balance = self.update_customer_balance(data_headers, **data)
            if bip_balance["status"] == 200:
                bundle_activate = self.client_ocs.post(
                    "/services/offers",
                    auth=(token, ""),
                    headers=data_headers,
                    json={"phonenumber": phonenumber, "offer_id": offer_id},
                )
                # verifier si l'activation du bundle s'est bien passee
                # sinon il faudra faire un rollback
                if bundle_activate["status"] == 200:
                    response = {
                        "status": 200,
                        "info": "activation reussie",
                        "content": params,
                        "code": "040-05-200",
                    }
                else:
                    due = -1 * int(amount)
                    self.rollback_ocs(token, phonenumber, params["headers"], due)
                    response = {
                        "status": 400,
                        "error": "echec activation",
                        "content": "un souci sur le charging systeme",
                    }
            else:
                response = {
                    "status": 400,
                    "error": "echec activation",
                    "content": "un souci sur le charging system :(",
                }
        except SyntaxError:
            return bad_request(message="parametres incomplets", code="040-05-400")
        LOGS.logger.info("response from OCS: {}".format(response))
        return response

    # achat d'une offre tv ou internet
    # via un partenaire
    def buying_offer_bluebase(self, **params):
        params["solde"] = params["amount"]
        params["version"] = "2"
        data_headers = params["headers"]
        token = params["token"]
        response = self.client_blue.post(
            "/ussd_buy_recharge", auth=(token, ""), headers=data_headers, json=params
        )
        LOGS.logger.info("params activation depuis bluebase: {}".format(params))
        LOGS.logger.info("resultat activation depuis bluebase: {}".format(response))
        return response

    # on utilise cette methode
    # pour l'activation
    # d'une offre partenaire
    def activate_offer_partner(self, **params):
        service_type = params["service_type"]
        if "offer_name" in params:
            offer_name = params["offer_name"]
        if service_type == "tv" or service_type == "internet":
            LOGS.logger.info(
                "activation {}:{}".format(params["service_type"], params["operator"])
            )
            result = self.buying_offer_bluebase(**params)
            if type(result) == tuple:
                result = result[0]

        if service_type == "bip" and offer_name == "credit_transfer_partner":
            LOGS.logger.info("patenaire: transfert de crédit, service_type:bip")
            result = self.recharging_bip_account(**params)

        if service_type == "bip" and offer_name == "buy_offer_partner":
            LOGS.logger.info("partenaire: achat d'un bundle, service_type:bip")
            result = self.buying_bip_bundle(**params)
        if type(result) == tuple:
            result = result[0]
        response = {"data": result["content"], "status": result["status"]}
        LOGS.logger.info(response)
        return response

    def activate_offer_bip(self, **params):
        # appel de l'api bluebase
        # pour activer une offre TV ou Internet
        # l'activation via un partenaire
        # ne passe pas par ocs
        data_headers = params["headers"]
        LOGS.logger.info(params)
        token = params["token"]
        if params["operator"] == "bip":
            billing_state = self.billing_ocs(**params)
            if billing_state["status"] == 200:
                # faire connaitre à bluebase le prix de l'offre
                params["solde"] = params["amount"]
                result = self.client_blue.post(
                    "/ussd_buy_recharge",
                    auth=(token, ""),
                    headers=data_headers,
                    json=params,
                )
                LOGS.logger.info("reponse venant de 4D".format(result))
                if result["status"] == 200:
                    result = {
                        "status": 200,
                        "content": "activation reussie",
                        "headers": data_headers,
                    }
                # il y a des cas ou 4D envoi 500 ou 503
                else:
                    self.rollback_ocs(
                        token, params["caller_num"], params["headers"], params["amount"]
                    )
                    result = {
                        "status": 404,
                        "content": "echec activation",
                        "headers": data_headers,
                    }
            elif billing_state["status"] == 400:
                result = {
                    "status": 400,
                    "content": "echec facturation",
                    "headers": data_headers,
                }
            elif billing_state["status"] == 401:
                result = {
                    "status": 401,
                    "content": "solde insuffisant",
                    "headers": data_headers,
                }
        else:
            result = {
                "status": 500,
                "content": "erreur inconnue",
                "headers": data_headers,
            }
        LOGS.logger.info("activation TV ou Internet via bip")
        response = {"data": result["content"], "status": result["status"]}
        LOGS.logger.info(response)
        return response

    # Je dois renommer le nom de ma fonction
    def write_to_redis(self, **params):
        LOGS.logger.info(params)
        data_headers = params["headers"]
        request_id = data_headers["X-Request-Id"]
        redis_key = "request:ussd:{}".format(request_id)
        state = params["offer_state"]
        LOGS.logger.info(params)
        status = state["status"]
        # Definition des parametres à stocker dans la base redis
        # la liste des paramètres envoyés
        # est différente selon le service type
        # si l'un des param n'est pas inclus
        # On mettra sa valeur par defaut come "None"
        options = ["offer_ref", "bundle_name", "offer_name"]
        for p in options:
            if p not in params:
                params[p] = ""
        if "offer_name" in params and params["offer_name"] == "credit_transfer_partner":
            bundle_name = "mobile_money: achat_credit_bip"
            params["offer_ref"] = "000432"
        elif "offer_name" in params and params["offer_name"] == "buy_offer_partner":
            bundle_name = "mobile_money: {}".format(params["bundle_name"])
        else:
            bundle_name = params["bundle_name"]
        response = {
            "caller_num": params["caller_num"],
            "amount": params["amount"],
            "customer_id": params["customer_id"],
            "offer_ref": params["offer_ref"],
            "offer_name": params["offer_name"],
            "service_type": params["service_type"],
            "bundle_name": bundle_name,
            "operator": params["operator"],
            "device_name": params["device_id"],
            "transaction_date": params["transaction_date"],
            "request_id": params["request_id"],
            "partner_ref": params["partner_ref"],
        }
        # dependant de la valeur du status
        # On retournera ici les reponses
        if status == 200:
            response["code"] = "040-05-204"
            response["info"] = "activation reussie"
            response["status"] = status
            if response["service_type"] == "bip":
                # pour bip on ajoutera les infos dans une base postgres en plus
                # pour la compatibilisation cote odoo
                # on enregistrera que les activation abouties
                try:
                    write_bip_transaction_to_db(**response)
                except Exception as ex:
                    LOGS.logger.info("Un souci sur la base {}".format(ex))

        elif status == 504:
            response = {
                "code": "040-05-504",
                "error": "ressource indisponible",
                "message": "ressource indisponible",
                "status": 504,
            }
        else:
            response["code"] = "040-05-400"
            response["error"] = "echec activation"
            response["status"] = 400
        # On enregistre tout dans REDIS :)
        try:
            REQUEST_DATABASE.hmset(redis_key, response)
            return True
        except Exception as ex:
            LOGS.logger.info(ex)
            LOGS.logger.info("je teste au cas ou redis ne fonctionne pas")
            return False

    def notify_via_sms(self, **params):
        """
        dépendra de activate_offer
        dans le cas bip on envoi
        directement un sms pour
        le client
        """
        offer_name = params["bundle_name"]
        token = params["token"]
        service = params["service_type"]
        teny = self.locale(**params)
        # si on arrive pas à récupérer la
        # langue utilisée par le client
        # sur ocs, on mettra par defaut
        # fr
        if teny == "":
            teny = "fr"
        LOGS.logger.info(teny)
        result = params["offer_state"]
        LOGS.logger.info(result)
        if result["status"] == 200 or result["status"] == 204:
            translated_msg_1 = LOCALE_MSG[teny]["200_1"]
            translated_msg_2 = LOCALE_MSG[teny]["200_2"]
            message = "{} {} {} {}".format(
                translated_msg_1, service, offer_name, translated_msg_2
            )
        elif result["status"] == 401:
            translated_msg = LOCALE_MSG[teny]["401"]
            message = "{}".format(translated_msg)
        elif result["status"] == 503 or result["status"] == 404:
            translated_msg_1 = LOCALE_MSG[teny]["400_1"]
            translated_msg_2 = LOCALE_MSG[teny]["400_2"]

            message = "{} {}".format(translated_msg_1, translated_msg_2)

        elif result["status"] == 404:
            translated_msg_1 = LOCALE_MSG[teny]["400_1"]
            translated_msg_2 = LOCALE_MSG[teny]["400_2"]
            message = "{} {}".format(translated_msg_1, translated_msg_2)
            LOGS.logger.info("un souci sur 4D?")
        else:
            translated_msg_1 = LOCALE_MSG[teny]["400_1"]
            translated_msg_2 = LOCALE_MSG[teny]["400_2"]
            message = "{} {}".format(translated_msg_1, translated_msg_2)
        # une fois le message obtenu
        # on relaie le message à l'api de smsc
        # en mode asynchrone
        data_sms = {"msisdn": params["caller_num"], "message": message}
        LOGS.logger.info(message)
        result = self.client_smsc.post(
            "/sms", auth=(token, ""), headers=params["headers"], json=data_sms
        )
        if result["status"] != 200:
            LOGS.logger.critical("echec envoi sms")

    # utiliser payment au lieu de adjust account
    def billing_ocs(self, **params):
        caller_num = params["caller_num"]
        phonenumber = caller_num
        amount = params["amount"]
        data_headers = params["headers"]
        token = params["token"]
        if params["operator"] != "bip":
            add_infos = params["partner_ref"]
        else:
            add_infos = params["request_id"][:17]
        # récupération solde actuel
        try:
            account = self.client_ocs.get(
                "/subscribers/{phonenumber}/balances".format(phonenumber=phonenumber),
                auth=(token, ""),
                headers=data_headers,
                json={"account_type": "2000"},
            )
            LOGS.logger.info(account["content"]["message"])
            if account["status"] == 200:
                ocs_data = account["content"]["message"]
                balance = ocs_data["balance"]
            else:
                balance = 0
        except SyntaxError:
            return bad_request(message="parametres incomplets", code="040-05-400")
        # debiter ocs
        if int(balance) >= int(amount):
            # activate_offer
            # maj account ocs
            due = -1 * int(amount)
            new_balance = int(balance) + due
            LOGS.logger.info("masiso")
            account_new = self.client_ocs.put(
                "/subscribers/{phonenumber}/balances".format(phonenumber=phonenumber),
                auth=(token, ""),
                headers=data_headers,
                json={
                    "amount": due,
                    "operation_type": "2",
                    "account_type": 2000,
                    "infos": add_infos,
                },
            )
            # verifier si la requête s'est bien passée
            if account_new["status"] == 200 or account_new["status"] == 204:
                billing_state = {"status": 200, "balance": new_balance}
            else:
                billing_state = {"status": 400, "balance": balance}
        else:
            billing_state = {"status": 401, "balance": balance}
        return billing_state
        LOGS.logger.info(billing_state)

    # une fonction pour faire un rollback
    # pour ocs au cas ou l'activation n'a
    # pas réussi
    def rollback_ocs(self, token, caller_num, header_req, amount):
        phonenumber = caller_num
        account = self.client_ocs.put(
            "/subscribers/{phonenumber}/balances".format(phonenumber=phonenumber),
            auth=(token, ""),
            headers=header_req,
            json={"amount": int(amount), "operation_type": "2", "account_type": 2000},
        )
        if account["status"] == 200 or account["status"] == 204:
            LOGS.logger.info("rollback reussie: {}".format(account))
        else:
            LOGS.logger.critical("echec rollback: {}".format(account["content"]))

    # Récuperation langue
    def locale(self, **params):
        LOGS.logger.info(params)
        data = {}
        for key, value in params.items():
            data[key] = value
        lang = OCS().get_language(**data)
        if lang["status"] == 200:
            lang_code = lang["data"]["language_code"]
            if lang_code == 3:
                locale = "mg"
            elif lang_code == 2:
                locale = "fr"
            elif lang_code == 1:
                locale = "en"
        else:
            locale = ""
            LOGS.logger.info("un souci sur OCS,")
            LOGS.logger.info("un souci sur la recuperation du locale")
        return locale
